package org.hansken.plugin.extraction.runtime.grpc.client;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import org.hansken.plugin.extraction.api.RandomAccessData;
import org.hansken.plugin.extraction.api.SearchTrace;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientDataContext;

import static org.hansken.plugin.extraction.runtime.grpc.client.ExtractionPluginGrpcAdapter.CURRENT_PROCESS_TRACE_MARKER;
import static org.hansken.plugin.extraction.runtime.grpc.common.Checks.isMetaContext;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

/**
 * This class is used to read trace data. It keeps track of traces and dataTypes which are available and it keeps a cache
 * of RandomAccessData instances. These are created when read is called on the datastream for the first time.
 */
public class ExtractionPluginDataReader {
    // This map contains traces with data streams available to read
    private final Map<String, Set<String>> _dataAvailable = new HashMap<>(); // <traceID, <dataType1, dataType2>>
    // This map contains open RandomAccessData instances
    private final Map<String, Map<String, RandomAccessData>> _dataCache = new HashMap<>(); // <traceID, <dataType, data>>
    private final ClientDataContext _context;

    public ExtractionPluginDataReader(final ClientDataContext context) {
        _context = argNotNull("context", context);

        // Mark the data of the trace being extracted as available and add this data to the cache.
        if (!isMetaContext(_context)) {
            _dataCache
                .computeIfAbsent(CURRENT_PROCESS_TRACE_MARKER, k -> new HashMap<>())
                .put(_context.dataType(), _context.data());
            _dataAvailable.put(CURRENT_PROCESS_TRACE_MARKER, Set.of(_context.dataType()));
        }
    }

    /**
     * Mark the dataTypes of the provided trace as available.
     *
     * @param trace SearchTrace with available data
     */
    public void markDataAvailable(final SearchTrace trace) {
        _dataAvailable.put(trace.get("uid"), new HashSet<>(trace.getDataTypes()));
    }

    /**
     * Read count bytes of data starting on position from dataType of traceId.
     *
     * @param traceUid uid of the trace
     * @param dataType which type of data to read (html, raw, ...)
     * @param position offset to start reading
     * @param count number of bytes to read
     * @return byte array of available data
     * @throws IOException when reading data fails
     * @throws NoSuchElementException if no data for a given traceId and dataType combination is available
     */
    public byte[] read(final String traceUid, final String dataType, final long position, final int count) throws IOException {
        if (!_dataAvailable.containsKey(traceUid)) {
            throw new NoSuchElementException("No data available for trace with uid: " + traceUid);
        }
        final Map<String, RandomAccessData> dataMap = _dataCache.computeIfAbsent(traceUid, k -> new HashMap<>());
        if (!_dataAvailable.get(traceUid).contains(dataType)) {
            throw new NoSuchElementException("No data available for data type: " + dataType);
        }
        final RandomAccessData typedData = dataMap.computeIfAbsent(dataType, k -> _context.data(traceUid, dataType));

        typedData.seek(position);
        return typedData.readNBytes(count);
    }
}
