package org.hansken.plugin.extraction.runtime.grpc.client;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import org.hansken.plugin.extraction.api.Trace;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DataStreamTransferStateManager {
    private static final Logger LOG = LoggerFactory.getLogger(DataStreamTransferStateManager.class);

    private final Map<String, StreamTransferState> _transfers = new HashMap<>();
    private final Trace _trace;
    private final ExecutorService _executor;

    DataStreamTransferStateManager(final Trace trace, final ExecutorService executor) {
        _trace = trace;
        _executor = executor;
    }

    void start(final String dataType) throws IOException {
        if (_transfers.containsKey(dataType)) {
            throw new IllegalStateException("data stream transfer started for data stream '" + dataType + "', but was already started");
        }
        final StreamTransferState transfer = StreamTransferState.create(_trace, dataType, _executor);
        transfer.setup();
        transfer.start();
        _transfers.put(dataType, transfer);
    }

    StreamTransferState get(final String dataType) {
        assertStreamStarted(dataType);
        return _transfers.get(dataType);
    }

    void finish(final String dataType) throws IOException {
        assertStreamStarted(dataType);
        _transfers.remove(dataType).finish();
    }

    void assertStreamStarted(final String dataType) {
        if (!_transfers.containsKey(dataType)) {
            throw new IllegalStateException("data stream transfer not started for datastream '" + dataType + "'");
        }
    }

    void finishAll() {
        for (final String dataStream : _transfers.keySet()) {
            try {
                _transfers.remove(dataStream).finish();
            }
            catch (final Exception e) {
                LOG.error("error while trying to shut down transfer for datastream '{}'", dataStream, e);
            }
        }
    }
}
