package org.hansken.plugin.extraction.runtime.grpc.client;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.concurrent.TimeUnit.SECONDS;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.UncheckedIOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;

import org.hansken.plugin.extraction.api.Trace;

/**
 * Class representing the current state of a data stream being written to a {@link Trace}
 * (see {@link Trace#setData(String, InputStream)}). It encapsulates the writing of a data stream,
 * piping received chunks to an {@link InputStream} passed to the trace, and as such translates
 * between the server push-based model to the stream pull-based model.
 * <p>
 * This class can be reused to write data streams one by one on the same trace. It ensures only a single
 * data stream can be written at a time (as long as only a single transfer object is used per trace).
 * <p>
 * Example of basic flow:
 * <pre>{@code
 *     final ExecutorService executor = ...;
 *     final Trace trace = ...;
 *
 *     final StreamTransferState transfer = StreamTransferState.create(trace, executor);
 *     transfer.start("raw");
 *     transfer.write("raw", chunk);
 *     transfer.finish("raw");
 * }</pre>
 * <strong>Note:</strong> this implementation is not thread-safe.
 */
final class StreamTransferState {

    // TODO HANSKEN-14695: make buffer size configurable
    // 32 KiB buffer size
    private static final int DEFAULT_BUFFER_SIZE = 32 * 1024;
    // wait time for task to finish
    private static final long TASK_FINISH_WAIT_MS = SECONDS.toMillis(16);

    private final Trace _trace;
    private final ExecutorService _executor;
    private final AtomicBoolean _inTransfer;
    private final String _dataType;

    private OutputStream _stream;
    private Future<?> _task;

    private StreamTransferState(final Trace trace, final String dataType, final ExecutorService executor) {
        _trace = argNotNull("trace", trace);
        _executor = argNotNull("executor", executor);
        _inTransfer = new AtomicBoolean();
        _dataType = dataType;
    }

    /**
     * Initializes a new transfer object for writing data streams to given {@link Trace}. It will use given executor
     * for constructing the {@link Thread} in which data will be written to the {@link InputStream}.
     * <p>
     * The given {@link ExecutorService} will <i>never</i> be shut down by this implementation.
     *
     * @param trace the trace to write data to
     * @param dataType data type to write to
     * @param executor the thread factory
     * @return a new transfer instance
     */
    static StreamTransferState create(final Trace trace, final String dataType, final ExecutorService executor) {
        return new StreamTransferState(trace, dataType, executor);
    }

    void setup() throws IOException {
        final PipedInputStream input = new PipedInputStream(DEFAULT_BUFFER_SIZE);
        _stream = new PipedOutputStream(input);
        // executing in separate thread, because the incoming chunks of data from the gRPC server are
        // 'piped' to the given input stream, and we don't want to block
        _task = _executor.submit(() -> {
            try (input) {
                _trace.setData(_dataType, input);
            }
            catch (final IOException e) {
                throw new UncheckedIOException(e);
            }
        });
    }

    /**
     * Signal the start of writing a new data stream of given type.
     *
     * @return {@code this}
     * @throws IOException when an I/O error occurs
     * @throws IllegalStateException if {@code this} is in a started state
     */
    StreamTransferState start() throws IOException {
        flagStart();
        return this;
    }

    /**
     * Write a chunk of data to the data stream. The data type is used as a sanity check,
     * in order to validate we are writing to the correct stream.
     *
     * @param data the chunk of data to write
     * @return {@code this}
     * @throws IOException when an I/O error occurs
     * @throws IllegalStateException if {@code this} is not in a started state, or if the data type does not match
     */
    StreamTransferState write(final byte[] data) throws IOException {
        assertInitialized();
        _stream.write(data);
        return this;
    }

    /**
     * Signal the end of writing the data stream of given type. Will wait for flush of any previously written chunks.
     * The data type is used as a sanity check, in order to validate we are closing the correct stream. After this call,
     * a new transfer can be initiated using {@link #start()}.
     *
     * @return {@code this}
     * @throws IOException when an I/O error occurs
     * @throws IllegalStateException if {@code this} is not in a started state, if the data type does not match
     *                               or if an error occurs awaiting the write of the remaining data
     */
    StreamTransferState finish() throws IOException {
        assertInitialized();
        shutdownAndAwait();
        return this;
    }

    /**
     * Terminate this transfer. Will wait for flush of any previously written chunks. After
     * calling shutdown, any usage of this instance will result in undefined behaviour.
     * <p>
     * This function should be called in case of error, in order to close the running transfer.
     *
     * @throws IOException when an I/O error occurs
     * @throws IllegalStateException when an error occurs awaiting the write of the remaining data
     */
    void shutdownAndAwait() throws IOException {
        if (!_inTransfer.get()) {
            return;
        }
        try {
            _stream.flush();
            // close signals EOF to the InputStream, so after finishing reading the rest of the buffer,
            // the InputStream reading will stop blocking and the Thread will be allowed to finish
            _stream.close();
            // wait until write to the data stream is fully finished and the Thread finishes
            _task.get(TASK_FINISH_WAIT_MS, MILLISECONDS);
        }
        catch (final InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException(e);
        }
        catch (final TimeoutException | ExecutionException e) {
            throw new IllegalStateException(e);
        }
        finally {
            cleanup();
            flagFinish();
        }
    }

    private void cleanup() {
        _task = null;
        _stream = null;
    }

    private void flagStart() {
        if (_inTransfer.getAndSet(true)) {
            throw new IllegalStateException("already writing stream of type: " + _dataType);
        }
    }

    private void flagFinish() {
        _inTransfer.set(false);
    }

    private void assertInitialized() {
        if (!_inTransfer.get()) {
            throw new IllegalStateException("currently not in transfer mode for stream of type: " + _dataType);
        }
    }
}
