package org.hansken.plugin.extraction.runtime.grpc.client.api;

import java.io.IOException;
import java.util.Map;

import org.hansken.plugin.extraction.api.BaseExtractionPlugin;
import org.hansken.plugin.extraction.api.DataContext;
import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.Trace;
import org.hansken.plugin.extraction.api.TraceSearcher;
import org.hansken.plugin.extraction.api.TransformerArgument;
import org.hansken.plugin.extraction.api.TransformerLabel;

/**
 * {@link ExtractionPlugin} for client side implementations. Compared to other extraction plugins,
 * it expects a specific types of {@link Trace} being processed
 * (see {@link #process(ClientTrace, ClientDataContext)} and
 * (see {@link #processDeferred(ClientTrace, ClientDataContext, TraceSearcher)}.
 * <p>
 * The reason for this is that it is used to simplify the implementation for creating children over
 * a gRPC network. See {@link ClientTrace} for more information.
 */
public interface RemoteExtractionPlugin extends BaseExtractionPlugin {

    /**
     * Checks that the API implemented by the remote plugin is compatible with the currently loaded (runtime) Extraction Plugin SDK version.
     *
     * @return true if compatible, false otherwise
     */
    boolean isCompatible();

    /**
     * See {@link ExtractionPlugin#process(Trace, DataContext)}, but requires a special type of {@link Trace}.
     *
     * @param trace the trace to process
     * @param dataContext data context of this trace extraction
     * @throws IOException when an I/O error occurs
     */
    void process(ClientTrace trace, ClientDataContext dataContext) throws IOException;

    /**
     * See {@link ExtractionPlugin#process(Trace, DataContext)}, but requires a special type of {@link Trace}.
     *
     * @param trace the trace to process
     * @param dataContext data context of this trace extraction
     * @param searcher the trace searcher
     * @throws IOException when an I/O error occurs
     */
    void processDeferred(ClientTrace trace, ClientDataContext dataContext, TraceSearcher searcher) throws IOException;

    /**
     * Calls a transformer method of a remote extraction plugin.
     *
     * @param label Describes the name, parameters and return type of the transformer method that we want to invoke.
     * @param arguments The actual arguments we want to pass to the method. Specified as a name and value pair.
     * @return Returns a value that was produced by the transformer method.
     */
    TransformerArgument transform(TransformerLabel label, Map<String, String> arguments);
}
