package org.hansken.plugin.extraction.runtime.grpc.client;

import static java.util.Collections.singletonList;

import static org.hansken.extraction.plugin.grpc.ExtractionPluginServiceGrpc.SERVICE_NAME;

import java.time.Duration;
import java.util.List;
import java.util.Map;

/**
 * A configurable exponential backoff policy which can be used by an {@link ExtractionPluginClient}
 * for trying to connect when requesting the plugin information or processing a Trace,
 * instead of failing directly due to unavailability.
 * <p>
 * The retry algorithm will at each iteration wait between 0 and the current iteration backoff
 * time. The backoff starts with the {@link #initialBackoffMs(Duration)}, and is multiplied
 * each iteration using {@link #backOffMultiplier(double)}, up to {@link #maxBackOff(Duration)}.
 */
public final class RetryPolicy {

    private int _maxAttempts;
    private Duration _initialBackoff;
    private Duration _maxBackOff;
    private double _backOffMultiplier;

    /**
     * Get a default policy, which has the following parameters:
     * {@code
     * maxAttempts=10,
     * initialBackoff=250ms,
     * maxBackOff=10s,
     * backOffMultiplier=2
     * }.
     *
     * @return a retry policy with default settings
     */
    public static RetryPolicy withDefaultSettings() {
        return new RetryPolicy()
            .maxAttempts(10)
            .initialBackoffMs(Duration.ofMillis(250))
            .maxBackOff(Duration.ofSeconds(10))
            .backOffMultiplier(2);
    }

    int maxAttempts() {
        return _maxAttempts;
    }

    /**
     * Set the maximum total attempts trying to connect. The
     * value should be at least 2.
     *
     * @param maxAttempts the maximum number of attempts
     * @return {@code this}
     */
    public RetryPolicy maxAttempts(final int maxAttempts) {
        _maxAttempts = maxAttempts;
        return this;
    }

    /**
     * The initial wait time bound of the first retry iteration. Must be less than
     * or equal to the {@link #maxBackOff(Duration) maximum backoff}.
     *
     * @param initialBackoff the initial backoff value
     * @return {@code this}
     */
    public RetryPolicy initialBackoffMs(final Duration initialBackoff) {
        _initialBackoff = initialBackoff;
        return this;
    }

    /**
     * The maximum wait time bound for any iteration. Must be greater than
     * or equal to the {@link #initialBackoffMs(Duration) initial backoff}.
     *
     * @param maxBackOff the maximum backoff value
     * @return {@code this}
     */
    public RetryPolicy maxBackOff(final Duration maxBackOff) {
        _maxBackOff = maxBackOff;
        return this;
    }

    /**
     * The factor to multiply the maximum backoff value with on each iteration. Should
     * be greater than or equal to 1.
     *
     * @param backOffMultiplier the backoff scale factor
     * @return {@code this}
     */
    public RetryPolicy backOffMultiplier(final double backOffMultiplier) {
        _backOffMultiplier = backOffMultiplier;
        return this;
    }

    Map<String, Object> toMethodConfigMap() {
        return Map.of("methodConfig", List.of(
            Map.of(
                "name", List.of(
                    Map.of(
                        "service", SERVICE_NAME,
                        "method", "pluginInfo"),
                    Map.of(
                        "service", SERVICE_NAME,
                        "method", "process")
                ),
                "retryPolicy", Map.of(
                    "maxAttempts", (double) _maxAttempts,
                    "initialBackoff", (_initialBackoff.toMillis() / 1000.0) + "s",
                    "maxBackoff", (_maxBackOff.toMillis() / 1000.0) + "s",
                    "backoffMultiplier", _backOffMultiplier,
                    "retryableStatusCodes", singletonList("UNAVAILABLE")
                ))
            )
        );
    }
}
