package org.hansken.plugin.extraction.runtime.grpc.common;

import static java.lang.String.format;
import static java.time.ZoneOffset.UTC;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Locale.ROOT;
import static java.util.stream.Collectors.toList;

import static org.hansken.plugin.extraction.runtime.grpc.common.VersionUtil.LAST_VERSION_TO_SUPPORT_PLUGIN_INFO_NAME;
import static org.hansken.plugin.extraction.runtime.grpc.common.VersionUtil.MINIMAL_SUPPORTED_VERSION;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;

import org.hansken.extraction.plugin.grpc.RpcAuthor;
import org.hansken.extraction.plugin.grpc.RpcBoolean;
import org.hansken.extraction.plugin.grpc.RpcBytes;
import org.hansken.extraction.plugin.grpc.RpcDouble;
import org.hansken.extraction.plugin.grpc.RpcDoubleList;
import org.hansken.extraction.plugin.grpc.RpcEmptyList;
import org.hansken.extraction.plugin.grpc.RpcEmptyMap;
import org.hansken.extraction.plugin.grpc.RpcFinish;
import org.hansken.extraction.plugin.grpc.RpcInteger;
import org.hansken.extraction.plugin.grpc.RpcIsoDateString;
import org.hansken.extraction.plugin.grpc.RpcLatLong;
import org.hansken.extraction.plugin.grpc.RpcList;
import org.hansken.extraction.plugin.grpc.RpcLong;
import org.hansken.extraction.plugin.grpc.RpcLongList;
import org.hansken.extraction.plugin.grpc.RpcMap;
import org.hansken.extraction.plugin.grpc.RpcMaturity;
import org.hansken.extraction.plugin.grpc.RpcNull;
import org.hansken.extraction.plugin.grpc.RpcPluginIdentifier;
import org.hansken.extraction.plugin.grpc.RpcPluginInfo;
import org.hansken.extraction.plugin.grpc.RpcPluginResources;
import org.hansken.extraction.plugin.grpc.RpcPluginType;
import org.hansken.extraction.plugin.grpc.RpcRangedTransformation;
import org.hansken.extraction.plugin.grpc.RpcSearchScope;
import org.hansken.extraction.plugin.grpc.RpcStart;
import org.hansken.extraction.plugin.grpc.RpcString;
import org.hansken.extraction.plugin.grpc.RpcStringList;
import org.hansken.extraction.plugin.grpc.RpcStringMap;
import org.hansken.extraction.plugin.grpc.RpcTracelet;
import org.hansken.extraction.plugin.grpc.RpcTransformation;
import org.hansken.extraction.plugin.grpc.RpcTransformer;
import org.hansken.extraction.plugin.grpc.RpcTransformerArgument;
import org.hansken.extraction.plugin.grpc.RpcTransformerResponse;
import org.hansken.extraction.plugin.grpc.RpcUnixTime;
import org.hansken.extraction.plugin.grpc.RpcVector;
import org.hansken.extraction.plugin.grpc.RpcZonedDateTime;
import org.hansken.plugin.extraction.api.Author;
import org.hansken.plugin.extraction.api.LatLong;
import org.hansken.plugin.extraction.api.MaturityLevel;
import org.hansken.plugin.extraction.api.PluginId;
import org.hansken.plugin.extraction.api.PluginInfo;
import org.hansken.plugin.extraction.api.PluginResources;
import org.hansken.plugin.extraction.api.PluginType;
import org.hansken.plugin.extraction.api.Trace;
import org.hansken.plugin.extraction.api.TraceSearcher.SearchScope;
import org.hansken.plugin.extraction.api.TransformerLabel;
import org.hansken.plugin.extraction.api.Vector;
import org.hansken.plugin.extraction.api.transformations.DataRange;
import org.hansken.plugin.extraction.api.transformations.DataTransformation;
import org.hansken.plugin.extraction.api.transformations.RangedDataTransformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;

/**
 * Utility class to convert from gRPC messages to their native Java or plugin API counterparts.
 * For the other way around, see {@link Pack}.
 */
public final class Unpack {
    private static final Logger LOG = LoggerFactory.getLogger(Unpack.class);

    private static final Unpacker PRIMITIVE_UNPACKER = Unpacker.create()
        .register(RpcNull.class, any -> null)
        .register(RpcBytes.class, any -> any.getValue().toByteArray())
        .register(RpcBoolean.class, RpcBoolean::getValue)
        .register(RpcInteger.class, RpcInteger::getValue)
        .register(RpcLong.class, RpcLong::getValue)
        .register(RpcDouble.class, RpcDouble::getValue)
        .register(RpcString.class, RpcString::getValue)
        .register(RpcUnixTime.class, any -> ZonedDateTime.ofInstant(Instant.ofEpochMilli(any.getValue()), UTC))
        .register(RpcVector.class, Unpack::unpackVector)
        .register(RpcEmptyList.class, any -> emptyList())
        .register(RpcEmptyMap.class, any -> emptyMap())
        .register(RpcStringList.class, RpcStringList::getValuesList)
        .register(RpcStringMap.class, RpcStringMap::getEntriesMap)
        .register(RpcMap.class, Unpack::getGetEntriesMap)
        .register(RpcLongList.class, RpcLongList::getValuesList)
        .register(RpcDoubleList.class, RpcDoubleList::getValuesList)
        .register(RpcLatLong.class, any -> LatLong.of(any.getLatitude(), any.getLongitude()))
        .register(RpcZonedDateTime.class, any -> {
            final ZoneOffset offset = ZoneOffset.of(any.getZoneOffset());
            final ZoneId id = ZoneId.of(any.getZoneId());
            return ZonedDateTime.ofInstant(
                LocalDateTime.ofEpochSecond(any.getEpochSecond(), any.getNanoOfSecond(), offset), offset, id
            );
        })
        .register(RpcIsoDateString.class, any -> ZonedDateTime.parse(any.getValue(), DateTimeFormatter.ISO_ZONED_DATE_TIME));

    private Unpack() {
    }

    /**
     * Unpacks an RpcVector into a Vector.
     * @param vector The RpcVector to be unpacked.
     * @return The unpacked Vector.
     */
    private static Vector unpackVector(final RpcVector vector) {
        return Vector.asVector(vector.getValue().toByteArray());
    }

    private static Map<String, Object> getGetEntriesMap(final RpcMap rpcMap) {
        final Map<String, Object> map = new HashMap<>();
        rpcMap.getEntriesMap().forEach((key, value) -> map.put(key, Unpack.primitive(value)));
        return map;
    }

    /**
     * Unpack an {@link Any} containing an {@link RpcStart} message.
     *
     * @param any the any to unpack
     * @return the start message
     * @throws IllegalArgumentException if the any does not contain a message of type {@link RpcStart}
     */
    public static RpcStart start(final Any any) {
        return Unpack.any(any, RpcStart.class);
    }

    /**
     * Unpack an {@link Any} containing an {@link RpcFinish} message.
     *
     * @param any the any to unpack
     * @return the finish message
     * @throws IllegalArgumentException if the any does not contain a message of type {@link RpcFinish}
     */
    public static RpcFinish finish(final Any any) {
        return Unpack.any(any, RpcFinish.class);
    }

    /**
     * Upack an {@link Any} containing a message of given message type.
     *
     * @param any the any to unpack
     * @param messageType class of the message contained in the any
     * @param <T> the type of the message contained in the any
     * @return the message
     * @throws IllegalArgumentException if the any does not contain a message of given type
     */
    public static <T extends Message> T any(final Any any, final Class<T> messageType) {
        try {
            return any.unpack(messageType);
        }
        catch (final InvalidProtocolBufferException e) {
            throw new IllegalArgumentException("unhandled type of message: " + any.getTypeUrl() + ", expected " + messageType.getCanonicalName());
        }
    }

    /**
     * Convert given primitive message to their native Java counterpart.
     *
     * @param any the any container of the primitive message
     * @return the unpacked primitive
     * @throws IllegalArgumentException if the any is not of an expected primitive
     *                                  message type
     */
    public static Object primitive(final Any any) {
        return PRIMITIVE_UNPACKER.unpack(any);
    }

    /**
     * Convert primitive {@link RpcBytes} message to the native Java {@code byte[]} counterpart.
     *
     * @param any the any container of the primitive rpc bytes message
     * @return the packed bytes
     * @throws IllegalArgumentException if given any does not contain an RpcBytes message
     */
    public static byte[] bytes(final Any any) {
        return Unpack.any(any, RpcBytes.class).getValue().toByteArray();
    }

    /**
     * Convert given {@link RpcPluginInfo} to their {@link PluginInfo} counterpart.
     * <p>
     * The API version that is available in {@link RpcPluginInfo} is not returned in the {@link PluginInfo} instance,
     * to get this value use {@link #pluginApiVersion(RpcPluginInfo)}.
     *
     * @param rpcPluginInfo the plugin info to convert
     * @return the converted info
     * @throws IllegalArgumentException when a property of given info is invalid as per the
     *                                  {@link PluginInfo.Builder} methods
     */
    public static PluginInfo pluginInfo(final RpcPluginInfo rpcPluginInfo) {
        final PluginInfo.Builder builder = PluginInfo.builderFor(Unpack.pluginType(rpcPluginInfo.getType()))
            .pluginVersion(rpcPluginInfo.getVersion())
            .description(rpcPluginInfo.getDescription())
            .author(Unpack.author(rpcPluginInfo.getAuthor()))
            .maturityLevel(Unpack.maturityLevel(rpcPluginInfo.getMaturity()))
            .hqlMatcher(rpcPluginInfo.getMatcher())
            .webpageUrl(rpcPluginInfo.getWebpageUrl())
            .deferredIterations(rpcPluginInfo.getDeferredIterations())
            .resources(pluginResources(rpcPluginInfo.getResources()));
        // If name is empty, version 0.4.0 or later is used, because from version 0.4.0 Pack won't set the name,
        // and in versions prior to 0.4.0, name cannot be empty. From version 0.4.0, the pluginId is used.
        if (isVersionOlderThan040(rpcPluginInfo)) {
            builder.name(rpcPluginInfo.getName());
            LOG.debug("No plugin id present. Incoming message was created by SDK version " + rpcPluginInfo.getApiVersion());
        }
        else {
            builder.id(pluginId(rpcPluginInfo.getId()))
                .license(rpcPluginInfo.getLicense());
        }
        if (rpcPluginInfo.getTransformersList() != null && !rpcPluginInfo.getTransformersList().isEmpty()) {
            builder.transformer(transformers(rpcPluginInfo.getTransformersList()));
        }
        return builder.build();
    }

    public static PluginId pluginId(final RpcPluginIdentifier rpcPluginIdentifier) {
        return new PluginId(rpcPluginIdentifier.getDomain(),
            rpcPluginIdentifier.getCategory(),
            rpcPluginIdentifier.getName());
    }

    public static List<TransformerLabel> transformers(final List<RpcTransformer> transformers) {
        return transformers
            .stream()
            .map(t -> new TransformerLabel(t.getMethodName(), t.getParametersMap(), t.getReturnType()))
            .collect(toList());
    }

    /**
     * Returns the API version contained in the {@link RpcPluginInfo} instance.
     *
     * @param rpcPluginInfo the plugin info that contains the API version
     * @return API version
     */
    public static String pluginApiVersion(final RpcPluginInfo rpcPluginInfo) {
        return rpcPluginInfo.getApiVersion();
    }

    /**
     * Convert given {@link RpcTracelet} to their {@link Trace.Tracelet} counterpart.
     *
     * @param rpcTracelet the info to convert
     * @return the converted info
     * @throws IllegalArgumentException when a property of given info is invalid as per the
     *                                  {@link PluginInfo.Builder} methods
     */
    public static Trace.Tracelet tracelet(final RpcTracelet rpcTracelet) {
        return new Trace.Tracelet(rpcTracelet.getName(),
            getTraceletProperties(rpcTracelet));
    }

    private static List<Trace.TraceletProperty> getTraceletProperties(final RpcTracelet property) {
        final List<Trace.TraceletProperty> traceletPropertiesValue = new ArrayList<>();
        property.getPropertiesList().forEach(traceletProperty -> {
            traceletPropertiesValue.add(new Trace.TraceletProperty(traceletProperty.getName(), Unpack.primitive(traceletProperty.getValue())));
        });
        return traceletPropertiesValue;
    }

    /**
     * Convert given {@link RpcAuthor} to their {@link Author} counterpart.
     *
     * @param author the author to convert
     * @return the converted author
     * @throws IllegalArgumentException when a property of given author is invalid as per the
     *                                  {@link Author.Builder} methods
     */
    public static Author author(final RpcAuthor author) {
        return Author.builder()
            .name(author.getName())
            .email(author.getEmail())
            .organisation(author.getOrganisation())
            .build();
    }

    /**
     * Convert given {@link RpcMaturity} to their {@link MaturityLevel} counterpart.
     *
     * @param maturity the maturity to convert
     * @return the converted maturity
     * @throws IllegalArgumentException when the given maturity is not mappable
     */
    public static MaturityLevel maturityLevel(final RpcMaturity maturity) {
        final int number = maturity.getNumber();
        if (number > MaturityLevel.values().length) {
            throw new IllegalArgumentException(format(ROOT, "maturity level %s has a too large number: %d", maturity, number));
        }
        return MaturityLevel.values()[number];
    }

    /**
     * Retrieve the response value from {@link RpcTransformerResponse}.
     * Currently only primitives are supported as defined in TransformerArgument.allowedTypes.
     *
     * @param response The {@link RpcTransformerResponse} that we want to retrieve the response value from.
     * @return The response value that is stored within the {@link RpcTransformerResponse}}.
     */
    public static RpcTransformerArgument transformerResponse(final RpcTransformerResponse response) {
        return response.getResponse();
    }

    /**
     * Unpacks a generic non-homogenous map into a native Java {@link Map}.
     * @param map The {@link RpcMap} that should be unpacked.
     * @return The unpacked map.
     */
    public static Map<String, Object> genericMap(final RpcMap map) {
        final Map<String, Object> convertedMap = new HashMap<>();

        for (Map.Entry<String, Any> entry : map.getEntriesMap().entrySet()) {
            try {
                final String key = entry.getKey();
                final Any value = entry.getValue();
                if (value.is(RpcList.class)) {
                    convertedMap.put(key, genericList(value.unpack(RpcList.class)));
                }
                else if (entry.getValue().is(RpcMap.class)) {
                    convertedMap.put(key, genericMap(value.unpack(RpcMap.class)));
                }
                else {
                    convertedMap.put(key, PRIMITIVE_UNPACKER.unpack(value));
                }
            } catch (final InvalidProtocolBufferException e) {
                throw new RuntimeException("Failed to unpack Any value", e);
            }
        }

        return convertedMap;
    }

    /**
     * Unpacks a generic non-homogenous list into a native Java {@link List}.
     * @param list The {@link RpcList} that should be unpacked.
     * @return The unpacked list.
     */
    public static List<Object> genericList(final RpcList list) {
        final List<Object> convertedList = new ArrayList<>();

        for (Any value : list.getValuesList()) {
            try {
                if (value.is(RpcList.class)) {
                    convertedList.add(genericList(value.unpack(RpcList.class)));
                }
                else if (value.is(RpcMap.class)) {
                    convertedList.add(genericMap(value.unpack(RpcMap.class)));
                }
                else {
                    convertedList.add(PRIMITIVE_UNPACKER.unpack(value));
                }
            } catch (final InvalidProtocolBufferException e) {
                throw new RuntimeException("Failed to unpack Any value", e);
            }
        }

        return convertedList;
    }

    /**
     * Convert a {@link RpcTransformerArgument} into a Java native type.
     * Currently only primitives are supported as defined in PRIMITIVE_UNPACKER.
     *
     * @param response The {@link RpcTransformerArgument} that we want to unpack into a Java native value
     * @return The response value that is stored within the {@link RpcTransformerResponse}}.
     */
    public static Object transformerArgument(final RpcTransformerArgument response) {
        switch (response.getTypeCase()) {
            case BOOLEAN:
                return PRIMITIVE_UNPACKER.unpack(Any.pack(response.getBoolean()));
            case BYTES:
                return PRIMITIVE_UNPACKER.unpack(Any.pack(response.getBytes()));
            case LONG:
                return PRIMITIVE_UNPACKER.unpack(Any.pack(response.getLong()));
            case DOUBLE:
                return PRIMITIVE_UNPACKER.unpack(Any.pack(response.getDouble()));
            case STRING:
                return PRIMITIVE_UNPACKER.unpack(Any.pack(response.getString()));
            case VECTOR:
                return PRIMITIVE_UNPACKER.unpack(Any.pack(response.getVector()));
            case LATLONG:
                return PRIMITIVE_UNPACKER.unpack(Any.pack(response.getLatLong()));
            case DATETIME:
                return PRIMITIVE_UNPACKER.unpack(Any.pack(response.getDatetime()));
            case LIST:
                return genericList(response.getList());
            case MAP:
                return genericMap(response.getMap());
            case TYPE_NOT_SET:
                throw new IllegalArgumentException("Type not set in RpcTransformerArgument");
            default:
                throw new IllegalStateException("Unexpected value: " + response.getTypeCase());
        }
    }

    /**
     * Convert given {@link RpcPluginType} to their {@link PluginType} counterpart.
     *
     * @param type the type to convert
     * @return the converted type
     * @throws IllegalArgumentException when the given type is not mappable
     */
    public static PluginType pluginType(final RpcPluginType type) {
        final int number = type.getNumber();
        if (number > PluginType.values().length) {
            throw new IllegalArgumentException(format(ROOT, "plugin type %s has a too large number: %d", type, number));
        }
        return PluginType.values()[number];
    }

    /**
     * Convert given {@link RpcPluginResources} to their {@link PluginResources} counterpart.
     *
     * @param rpcPluginResources the rpc plugin resources to convert
     * @return the converted plugin resources
     */
    public static PluginResources pluginResources(final RpcPluginResources rpcPluginResources) {
        final PluginResources.Builder builder = PluginResources.builder();
        Optional.of(rpcPluginResources.getMaxCpu()).ifPresent(builder::maximumCpu);
        Optional.of(rpcPluginResources.getMaxMemory()).ifPresent(builder::maximumMemory);
        Optional.of(rpcPluginResources.getMaxWorkers()).ifPresent(builder::maximumWorkers);

        return builder.build();
    }

    public static List<DataTransformation> transformations(final List<RpcTransformation> rpcDataTransformations) {
        return rpcDataTransformations.stream()
            .map(Unpack::transformation)
            .collect(toList());
    }

    private static DataTransformation transformation(final RpcTransformation rpcTransformation) {
        switch (rpcTransformation.getValueCase()) {
            case RANGEDTRANSFORMATION:
                final RpcRangedTransformation rpcRangedTransformation = rpcTransformation.getRangedTransformation();
                final List<DataRange> ranges = rpcRangedTransformation.getRangesList()
                    .stream()
                    .map(r -> new DataRange(r.getOffset(), r.getLength()))
                    .collect(toList());
                return new RangedDataTransformation(ranges);
            default:
                throw new IllegalStateException("Illegal transformation type: " + rpcTransformation.getValueCase().name());
        }
    }

    public static SearchScope searchScope(final RpcSearchScope scope) {
        if (scope == null) {
            return SearchScope.PROJECT;
        }
        switch (scope) {
            case Project:
                return SearchScope.PROJECT;
            case Image:
                return SearchScope.IMAGE;
            case UNRECOGNIZED:
            default:
                throw new IllegalArgumentException("unrecognized search scope: " + scope);
        }
    }

    private static boolean isVersionOlderThan040(final RpcPluginInfo rpcPluginInfo) {
        return VersionUtil.versionInRange(MINIMAL_SUPPORTED_VERSION, LAST_VERSION_TO_SUPPORT_PLUGIN_INFO_NAME, Version.fromString(rpcPluginInfo.getApiVersion()));
    }

    static final class Unpacker {

        private final Map<Class<? extends Message>, Function<Message, Object>> _unpackers = new HashMap<>();

        static Unpacker create() {
            return new Unpacker();
        }

        <T extends Message> Unpacker register(final Class<T> message, final Function<T, Object> unpacker) {
            _unpackers.put(message, (Function<Message, Object>) unpacker);
            return this;
        }

        Object unpack(final Any any) {
            try {
                for (final var unpacker : _unpackers.entrySet()) {
                    final Class<? extends Message> messageType = unpacker.getKey();
                    if (any.is(messageType)) {
                        return unpacker.getValue().apply(any.unpack(messageType));
                    }
                }
                throw new IllegalArgumentException("unhandled type of Any: " + any.getTypeUrl());
            }
            catch (final InvalidProtocolBufferException e) {
                throw new IllegalArgumentException(e);
            }
        }
    }
}
