package org.hansken.plugin.extraction.runtime.grpc.common;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotEmpty;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.util.Properties;

import org.hansken.plugin.extraction.api.PluginInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class that is used to return the current SDK (API) version,
 * and to test forward and backward compatibility.
 */
public final class VersionUtil {

    /**
     * The lowest version that is compatible with the current runtime version.
     * Note that if the lowest version is the next release version, please omit the SNAPSHOT-subversion.
     * <p>
     * Example: you are on 0.2.0-SNAPSHOT enter MINIMAL_SUPPORTED_VERSION as is 0.2.0
     */
    static final Version MINIMAL_SUPPORTED_VERSION = new Version(0, 2, 0);
    /**
     * {@link PluginInfo#name()} was deprecated from version <b>0.4.0</b> onwards.
     */
    static final Version LAST_VERSION_TO_SUPPORT_PLUGIN_INFO_NAME = Version.fromString("0.3.0");

    private static final Logger LOG = LoggerFactory.getLogger(VersionUtil.class);

    private static final Version RUNTIME_VERSION = Version.fromString(getApiVersion());

    private static String _apiVersion;

    private VersionUtil() {
        // utility class constructor
    }

    /**
     * Returns the extraction plugin API version.
     *
     * @return the extraction plugin API version
     */
    public static String getApiVersion() {
        if (_apiVersion == null) {
            _apiVersion = argNotEmpty("apiVersion", readApiVersion());
        }
        return _apiVersion;
    }

    /**
     * Utility method that compares a given version String for compatibility with the current SDK version.
     *
     * @param pluginVersionString Version-string of the remote plugin
     * @return true if plugin version API is compatible with the current SDK API version
     */
    public static boolean isCompatible(final String pluginVersionString) {
        // versions prior to 0.2.0 did not send any version information
        if (pluginVersionString == null || pluginVersionString.isEmpty()) {
            LOG.debug("plugin version is not compatible: plugin version information is missing");
            return false;
        }

        final Version pluginVersion = Version.fromString(pluginVersionString);

        if (minVersionIsCurrentSnapshot()) {
            // special case - this snapshot build sets the minimal API version to its release version
            // we'll have to downgrade the minimal version and plugin version to this current build version
            // so unit tests will pass in both the SNAPSHOT build AND the release build
            // TODO HANSKEN-15499: see if we can remove this special logic
            final Version pluginVersionAsSnapshot = new Version(pluginVersion.getMajorVersion(), pluginVersion.getMinorVersion(), pluginVersion.getPatchVersion(), RUNTIME_VERSION.getSubVersion());
            return versionInRange(RUNTIME_VERSION, RUNTIME_VERSION, pluginVersionAsSnapshot);
        }
        else {
            // normal case, this build (snapshot and release) is newer than the minimum version, just check versions :)
            return versionInRange(MINIMAL_SUPPORTED_VERSION, RUNTIME_VERSION, pluginVersion);
        }
    }

    /**
     * Utility method that compares two API versions.
     *
     * @param first  Version-string of a remote plugin API
     * @param second Version-string of a remote plugin API
     * @return true if {@code first} version is higher or equal than {@code second}
     */
    public static boolean isHigherOrEqualTo(final String first, final String second) {
        argNotEmpty("first", first);
        argNotEmpty("second", second);

        final Version firstVersion = Version.fromString(first);
        final Version secondVersion = Version.fromString(second);

        return firstVersion.isHigherOrEqualTo(secondVersion);
    }

    // package-private for unit testing
    static boolean versionInRange(final Version minVersion, final Version maxVersion, final Version pluginVersion) {
        // if (maxVersion <= pluginVersion <= minVersion)
        if (maxVersion.isHigherOrEqualTo(pluginVersion) && pluginVersion.isHigherOrEqualTo(minVersion)) {
            LOG.debug("plugin version (" + pluginVersion + ") is in range of compatible versions [" + minVersion + ", " + maxVersion + "]");
            return true;
        }

        LOG.debug("plugin version (" + pluginVersion + ") is not in range of compatible versions [" + minVersion + ", " + maxVersion + "]");
        return false;
    }

    private static boolean minVersionIsCurrentSnapshot() {
        return !RUNTIME_VERSION.getSubVersion().isEmpty()
            && MINIMAL_SUPPORTED_VERSION.equals(new Version(RUNTIME_VERSION.getMajorVersion(), RUNTIME_VERSION.getMinorVersion(), RUNTIME_VERSION.getPatchVersion()));
    }

    private static String readApiVersion() {
        try (InputStream resourceAsStream = VersionUtil.class.getResourceAsStream("/apiversion.properties")) {
            final Properties properties = new Properties();
            properties.load(resourceAsStream);
            return properties.getProperty("api.version");
        }
        catch (final IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}
