package org.hansken.plugin.extraction.runtime.grpc.common;

import static java.lang.Integer.parseInt;

/**
 * Helper class that represents an API version, semver compliant, in the form <code>major.minor.patch</code>
 * or <code>major.minor.patch-subversion</code>.
 */
class Version implements Comparable<Version> {
    private final int _majorVersion;
    private final int _minorVersion;
    private final int _patchVersion;
    private final String _subVersion;

    /**
     * Constructor of a semver (major.minor.patch) version.
     *
     * @param majorVersion Major part of version
     * @param minorVersion Minor part of version
     * @param patchVersion Patch part of version
     */
    Version(final int majorVersion, final int minorVersion, final int patchVersion) {
        this(majorVersion, minorVersion, patchVersion, "");
    }

    /**
     * Constructor of a semver (major.minor.patch) version with a subversion.
     *
     * @param majorVersion Major part of version
     * @param minorVersion Minor part of version
     * @param patchVersion Patch part of version
     * @param subVersion Subversion part of version
     */
    Version(final int majorVersion, final int minorVersion, final int patchVersion, final String subVersion) {
        _majorVersion = majorVersion;
        _minorVersion = minorVersion;
        _patchVersion = patchVersion;
        _subVersion = subVersion;
    }

    /**
     * Utility that converts a version string in the semantic versioning format to a {@link Version} instance.
     * <p>
     * Optionally, a version postfix (subVersion) can be provided after a - or + sign.
     * Version postfix is not taken into account when comparing Versions.
     * <p>
     * Examples:
     * <ul>
     *   <li>1.2.3</li>
     *   <li>1.2.3-SNAPSHOT</li>
     *   <li>1.2.3+SNAPSHOT</li>
     * </ul>
     *
     * @param version Version string to convert to {@link Version}
     * @return a {@link Version} representation of the input version string
     */
    static Version fromString(final String version) {
        // determine the start position of the subversion (if any)
        // subverpos is first - or + char, or else the full string length
        final String[] versionParts = version.split("-|\\+", 2);

        // determine the actual version parts
        final String[] semver = versionParts[0].split("\\.");
        if (semver.length != 3) {
            throw new IllegalArgumentException("got invalid version, not in the form major.minor.patch[-subversion], got: " + version);
        }
        final String subVersion = versionParts.length > 1 ? versionParts[1] : "";

        try {
            final int majorVersion = parseInt(semver[0]);
            final int minorVersion = parseInt(semver[1]);
            final int patchVersion = parseInt(semver[2]);
            return new Version(majorVersion, minorVersion, patchVersion, subVersion);
        }
        catch (final Exception e) {
            throw new IllegalArgumentException("got invalid version, could not parse numbers from: " + version);
        }
    }

    int getMajorVersion() {
        return _majorVersion;
    }

    int getMinorVersion() {
        return _minorVersion;
    }

    int getPatchVersion() {
        return _patchVersion;
    }

    String getSubVersion() {
        return _subVersion;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + _majorVersion;
        result = prime * result + _minorVersion;
        result = prime * result + _patchVersion;
        result = prime * result + ((_subVersion == null) ? 0 : _subVersion.hashCode());
        return result;
    }

    @Override
    public boolean equals(final Object other) {
        if (this == other) {
            return true;
        }
        if (!(other instanceof Version)) {
            return false;
        }
        return compareTo((Version) other) == 0;
    }

    @Override
    public int compareTo(final Version other) {
        // Returns a negative integer, zero, or a positive integer as this object is less than, equal to, or greater than the specified object.

        // as per comparable interface
        if (other == null) {
            throw new NullPointerException();
        }

        // first test major version
        if (_majorVersion != other._majorVersion) {
            return _majorVersion < other._majorVersion ? 1 : -1;
        }

        // same major version, test minor version
        if (_minorVersion != other._minorVersion) {
            return _minorVersion < other._minorVersion ? 1 : -1;
        }

        // same minor version, test patch version
        if (_patchVersion != other._patchVersion) {
            return _patchVersion < other._patchVersion ? 1 : -1;
        }

        // we do not take subversion checking into account, so 0.8.0 and 0.8.0-SNAPSHOT, and 0.8.0+SNAPSHOT are all equal
        // this allows for testing SNAPSHOT versions in Hansken for intended use
        return 0;
    }

    /**
     * Check if this version is higher or equal than another version.
     *
     * @param other the version to compare with
     * @return true if this version is higher or equal than {@code other}
     */
    public boolean isHigherOrEqualTo(final Version other) {
        return this.compareTo(other) <= 0;
    }

    @Override
    public String toString() {
        if (_subVersion.isEmpty()) {
            return _majorVersion + "." + _minorVersion + "." + _patchVersion;
        }
        return _majorVersion + "." + _minorVersion + "." + _patchVersion + "-" + _subVersion;
    }
}
