package org.hansken.plugin.extraction.runtime.grpc.common;

import static java.lang.String.format;
import static java.util.Locale.ROOT;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

import static org.hansken.plugin.extraction.runtime.grpc.common.Checks.isMetaContext;
import static org.hansken.plugin.extraction.runtime.grpc.common.VersionUtil.getApiVersion;

import java.io.IOException;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.hansken.extraction.plugin.grpc.RpcAuthor;
import org.hansken.extraction.plugin.grpc.RpcBoolean;
import org.hansken.extraction.plugin.grpc.RpcBytes;
import org.hansken.extraction.plugin.grpc.RpcDataContext;
import org.hansken.extraction.plugin.grpc.RpcDataStreamTransformation;
import org.hansken.extraction.plugin.grpc.RpcDouble;
import org.hansken.extraction.plugin.grpc.RpcDoubleList;
import org.hansken.extraction.plugin.grpc.RpcEmptyList;
import org.hansken.extraction.plugin.grpc.RpcEmptyMap;
import org.hansken.extraction.plugin.grpc.RpcEnrichTrace;
import org.hansken.extraction.plugin.grpc.RpcInteger;
import org.hansken.extraction.plugin.grpc.RpcLatLong;
import org.hansken.extraction.plugin.grpc.RpcList;
import org.hansken.extraction.plugin.grpc.RpcLong;
import org.hansken.extraction.plugin.grpc.RpcLongList;
import org.hansken.extraction.plugin.grpc.RpcMap;
import org.hansken.extraction.plugin.grpc.RpcMaturity;
import org.hansken.extraction.plugin.grpc.RpcNull;
import org.hansken.extraction.plugin.grpc.RpcPluginIdentifier;
import org.hansken.extraction.plugin.grpc.RpcPluginInfo;
import org.hansken.extraction.plugin.grpc.RpcPluginResources;
import org.hansken.extraction.plugin.grpc.RpcPluginType;
import org.hansken.extraction.plugin.grpc.RpcRandomAccessDataMeta;
import org.hansken.extraction.plugin.grpc.RpcRange;
import org.hansken.extraction.plugin.grpc.RpcRangedTransformation;
import org.hansken.extraction.plugin.grpc.RpcRead;
import org.hansken.extraction.plugin.grpc.RpcSearchRequest;
import org.hansken.extraction.plugin.grpc.RpcSearchResult;
import org.hansken.extraction.plugin.grpc.RpcSearchScope;
import org.hansken.extraction.plugin.grpc.RpcSearchTrace;
import org.hansken.extraction.plugin.grpc.RpcString;
import org.hansken.extraction.plugin.grpc.RpcStringList;
import org.hansken.extraction.plugin.grpc.RpcStringMap;
import org.hansken.extraction.plugin.grpc.RpcTrace;
import org.hansken.extraction.plugin.grpc.RpcTraceProperty;
import org.hansken.extraction.plugin.grpc.RpcTracelet;
import org.hansken.extraction.plugin.grpc.RpcTransformation;
import org.hansken.extraction.plugin.grpc.RpcTransformerArgument;
import org.hansken.extraction.plugin.grpc.RpcTransformerRequest;
import org.hansken.extraction.plugin.grpc.RpcUnixTime;
import org.hansken.extraction.plugin.grpc.RpcVector;
import org.hansken.extraction.plugin.grpc.RpcZonedDateTime;
import org.hansken.plugin.extraction.api.Author;
import org.hansken.plugin.extraction.api.DataContext;
import org.hansken.plugin.extraction.api.ImmutableTrace;
import org.hansken.plugin.extraction.api.LatLong;
import org.hansken.plugin.extraction.api.MaturityLevel;
import org.hansken.plugin.extraction.api.PluginId;
import org.hansken.plugin.extraction.api.PluginInfo;
import org.hansken.plugin.extraction.api.PluginResources;
import org.hansken.plugin.extraction.api.PluginType;
import org.hansken.plugin.extraction.api.RandomAccessData;
import org.hansken.plugin.extraction.api.SearchResult;
import org.hansken.plugin.extraction.api.SearchTrace;
import org.hansken.plugin.extraction.api.Trace;
import org.hansken.plugin.extraction.api.TraceSearcher.SearchScope;
import org.hansken.plugin.extraction.api.TransformerArgument;
import org.hansken.plugin.extraction.api.TransformerLabel;
import org.hansken.plugin.extraction.api.Vector;
import org.hansken.plugin.extraction.api.transformations.DataRange;
import org.hansken.plugin.extraction.api.transformations.DataTransformation;
import org.hansken.plugin.extraction.api.transformations.RangedDataTransformation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.protobuf.Any;
import com.google.protobuf.ByteString;
import com.google.protobuf.Message;

import io.grpc.Status.Code;
import io.grpc.StatusRuntimeException;

/**
 * Utility class to convert from native Java or plugin API data structures to their gRPC message counterparts.
 * For the other way around, see {@link Unpack}.
 */
public final class Pack {
    /**
     * Amount of bytes initially sent with the RpcRandomAccessDataMeta (for performance reasons).
     * It is useless to make it greater than the cache size.
     */
    public static final int FIRST_BYTES_SIZE = 1024 * 1024; // 1 Mb

    private static final Logger LOG = LoggerFactory.getLogger(Pack.class);

    private static final RpcNull NULL = RpcNull.getDefaultInstance();
    private static final RpcEmptyList EMPTY_LIST = RpcEmptyList.getDefaultInstance();
    private static final RpcEmptyMap EMPTY_MAP = RpcEmptyMap.getDefaultInstance();
    private static final Packer PRIMITIVE_PACKER = Packer.create()
        .register(byte[].class, object -> RpcBytes.newBuilder().setValue(ByteString.copyFrom(object)).build())
        .register(Boolean.class, object -> RpcBoolean.newBuilder().setValue(object).build())
        .register(Integer.class, object -> RpcInteger.newBuilder().setValue(object).build())
        .register(Long.class, object -> RpcLong.newBuilder().setValue(object).build())
        .register(Double.class, object -> RpcDouble.newBuilder().setValue(object).build())
        .register(String.class, object -> RpcString.newBuilder().setValue(object).build())
        .register(List.class, Pack::list)
        .register(Map.class, Pack::map)
        .register(Date.class, object -> RpcUnixTime.newBuilder().setValue(object.getTime()).build())
        .register(LatLong.class, object -> RpcLatLong.newBuilder()
            .setLatitude(object.latitude())
            .setLongitude(object.longitude())
            .build())
        .register(Vector.class, object -> RpcVector.newBuilder().setValue(ByteString.copyFrom(object.asBinary())).build())
        .register(ZonedDateTime.class, object -> RpcZonedDateTime.newBuilder()
            .setEpochSecond(object.toEpochSecond())
            .setNanoOfSecond(object.getNano())
            .setZoneOffset(object.getOffset().getId())
            .setZoneId(object.getZone().getId())
            .build());

    private Pack() {
    }

    /**
     * Convert given {@link Trace} to their {@link RpcTrace} counterpart.
     *
     * @param id the id of the trace
     * @param trace the trace to convert
     * @return the converted trace
     * @throws IllegalArgumentException if the trace contains properties which
     *                                  are not of an expected primitive type
     */
    public static RpcTrace trace(final String id, final ImmutableTrace trace) {
        final Set<RpcTraceProperty> rpcProperties = trace.properties().stream()
            .filter(filterAndLog(trace))
            .map(property -> Pack.property(property, trace.get(property)))
            .collect(toSet());

        return RpcTrace.newBuilder()
            .setId(id)
            .addAllTypes(trace.types())
            .addAllProperties(rpcProperties)
            .build();
    }

    /**
     * Convert given {@link SearchTrace} to their {@link RpcSearchTrace} counterpart.
     *
     * @param trace the search trace to convert
     * @return the converted trace
     * @throws IllegalArgumentException if the trace contains properties which
     *                                  are not of an expected primitive type
     */
    public static RpcSearchTrace searchTrace(final SearchTrace trace) {
        final Set<RpcTraceProperty> rpcProperties = trace.properties()
            .stream()
            .filter(filterAndLog(trace))
            .map(property -> property(property, trace.get(property)))
            .collect(toSet());
        final List<RpcRandomAccessDataMeta> data = trace.getDataTypes()
            .stream()
            .map(dataType -> randomAccessDataMeta(dataType, trace))
            .collect(toList());

        return RpcSearchTrace.newBuilder()
            .setId(trace.traceId())
            .addAllProperties(rpcProperties)
            .addAllData(data)
            .addAllTypes(trace.types()) // trace.types() and not the getDataTypes(), these are set on data itself!
            .build();
    }

    private static Predicate<? super String> filterAndLog(final ImmutableTrace trace) {
        return propertyKey -> {
            final Object propertyValue = trace.get(propertyKey);
            if (isSupportedByPacker(propertyValue)) {
                return true;
            }
            else {
                LOG.warn("Property {} is of type {} which isn't supported by the SDK..yet!", propertyKey, propertyValue.getClass().getName());
                return false;
            }
        };
    }

    private static boolean isSupportedByPacker(final Object object) {
        return PRIMITIVE_PACKER._packers.keySet().stream()
            .anyMatch(packerClass -> packerClass.isInstance(object));
    }

    public static RpcTraceProperty property(final String name, final Object value) {
        return RpcTraceProperty.newBuilder()
            .setName(name)
            .setValue(Any.pack(Pack.primitive(value)))
            .build();
    }

    /**
     * Convert given object to their RPC primitive message counterpart (see PrimitiveMessages proto definitions),
     * based on the runtime type. If the type is a generic {@link List} or {@link Map}, the elements, keys and values
     * must all be of the same type.
     *
     * @param object the object to convert
     * @return the converted object
     * @throws IllegalArgumentException if the type of the object is not supported,
     *                                  or if not all elements of a collection (like) object are of the same type,
     *                                  or if any of these elements is null
     */
    public static Message primitive(final Object object) {
        return PRIMITIVE_PACKER.pack(object);
    }

    @SuppressWarnings("unchecked")
    private static Message list(final List<?> list) {
        if (list.isEmpty()) {
            return EMPTY_LIST;
        }
        if (allOfType(list, String.class)) {
            return RpcStringList.newBuilder().addAllValues((List<String>) list).build();
        }
        if (allOfType(list, Integer.class)) {
            final List<Long> longList = ((List<Integer>) list).stream()
                .map(v -> (long) v)
                .collect(toList());
            return RpcLongList.newBuilder().addAllValues(longList).build();
        }
        if (allOfType(list, Long.class)) {
            return RpcLongList.newBuilder().addAllValues((List<Long>) list).build();
        }
        if (allOfType(list, Float.class)) {
            final List<Double> floatList = ((List<Float>) list).stream()
                .map(v -> (double) v)
                .collect(toList());
            return RpcDoubleList.newBuilder().addAllValues(floatList).build();
        }
        if (allOfType(list, Double.class)) {
            return RpcDoubleList.newBuilder().addAllValues((List<Double>) list).build();
        }
        throw new IllegalArgumentException("Only homogeneous List of type Strings, Integer, Long, Float, or Double are supported.");
    }

    private static Message map(final Map<?, ?> map) {
        if (map.isEmpty()) {
            return EMPTY_MAP;
        }
        // TODO remove in HANSKEN-15355 - leaving it for now, because the assumption is that RpcStringMap is faster than RpcMap
        if (allOfType(map, String.class)) {
            return RpcStringMap.newBuilder().putAllEntries((Map<String, String>) map).build();
        }
        final RpcMap.Builder mapBuilder = RpcMap.newBuilder();
        for (final Map.Entry<?, ?> entry : map.entrySet()) {
            final Any primitive = Any.pack(Pack.primitive(entry.getValue()));
            mapBuilder.putEntries((String) entry.getKey(), primitive);
        }

        return mapBuilder.build();
    }

    private static boolean allOfType(final List<?> list, final Class<?> type) {
        for (final Object element : list) {
            if (element == null) {
                throw new IllegalArgumentException("list must not contain a null");
            }
            if (!type.isInstance(element)) {
                return false;
            }
        }
        return true;
    }

    private static boolean allOfType(final Map<?, ?> map, final Class<?> type) {
        for (final var entry : map.entrySet()) {
            final Object key = entry.getKey();
            final Object value = entry.getValue();
            if (key == null || value == null) {
                throw new IllegalArgumentException("map must not contain a null key or value");
            }
            if (!type.isInstance(key) || !type.isInstance(value)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Convert given {@link DataContext} to their {@link RpcDataContext} counterpart.
     * Only the metadata of the data sequence will be packed, see {@link #metaOfData(String, RandomAccessData)}.
     *
     * @param context the context to convert
     * @return the converted context
     */
    public static RpcDataContext metaOfDataContext(final DataContext context) {
        return RpcDataContext.newBuilder()
            .setDataType(context.dataType())
            .setData(isMetaContext(context)
                // in case of a 'meta' context, there is no associated data
                ? RpcRandomAccessDataMeta.getDefaultInstance() : metaOfData(context.dataType(), context.data()))
            .build();
    }

    /**
     * Convert given {@link RandomAccessData} to their {@link RpcRandomAccessDataMeta} counterpart.
     * This does not copy the entire data contained in the sequence, only the {@link RandomAccessData#size() size}
     * and the first 1MB if available.
     *
     * @param dataType the dataType of the sequence
     * @param data the data sequence to convert
     * @return the converted data sequence
     */
    public static RpcRandomAccessDataMeta metaOfData(final String dataType, final RandomAccessData data) {
        final RpcRandomAccessDataMeta.Builder meta = RpcRandomAccessDataMeta.newBuilder()
            .setType(dataType)
            .setSize(data.size());
        try {
            data.seek(0);
            meta.setFirstBytes(ByteString.copyFrom(data.readNBytes(FIRST_BYTES_SIZE)));
        }
        catch (final IOException e) {
            LOG.warn("Exception occurred when setting first bytes to send.", e);
            // pass, don't fill the cache, we'll deal with it later...
        }
        return meta.build();
    }

    /**
     * Pack given dataType and trace into an RPC message. Note that this message does not include any data.
     *
     * @param dataType type of the data
     * @param trace trace used to infer size of the data
     * @return RpcRandomAccessDataMeta message
     */
    public static RpcRandomAccessDataMeta randomAccessDataMeta(final String dataType, final ImmutableTrace trace) {
        // data.dataType.size is a long property, but the test framework is unaware of the trace model, so it returns an int.
        final long size = ((Number) trace.get("data." + dataType + ".size")).longValue();
        final RpcRandomAccessDataMeta.Builder meta = RpcRandomAccessDataMeta.newBuilder()
            .setType(dataType)
            .setSize(size);
        return meta.build();
    }

    /**
     * Convert given {@link PluginInfo} to their {@link RpcPluginInfo} counterpart.
     *
     * @param info the info to convert
     * @return the converted info
     */
    public static RpcPluginInfo pluginInfo(final PluginInfo info) {
        final RpcPluginInfo.Builder builder = RpcPluginInfo.newBuilder()
            // don't send name, it is deprecated, setId should be used instead
            .setApiVersion(getApiVersion())
            .setType(Pack.pluginType(info.pluginType()))
            .setVersion(info.pluginVersion())
            .setDescription(info.description())
            .setAuthor(Pack.author(info.author()))
            .setMaturity(Pack.maturityLevel(info.maturityLevel()))
            .setMatcher(info.hqlMatcher())
            .setWebpageUrl(info.webpageUrl())
            .setDeferredIterations(info.deferredIterations())
            .setId(pluginId(info.id()))
            .setLicense(info.license());

        if (info.resources() != null) {
            builder.setResources(pluginResources(info.resources()));
        }

        return builder.build();
    }

    public static RpcPluginIdentifier pluginId(final PluginId id) {
        return RpcPluginIdentifier.newBuilder()
            .setDomain(id.domain())
            .setCategory(id.category())
            .setName(id.name())
            .build();
    }

    /**
     * Convert given {@link Author} to their {@link RpcAuthor} counterpart.
     *
     * @param author the author to convert
     * @return the converted author
     */
    public static RpcAuthor author(final Author author) {
        return RpcAuthor.newBuilder()
            .setName(author.name())
            .setEmail(author.email())
            .setOrganisation(author.organisation())
            .build();
    }

    /**
     * Convert given {@link MaturityLevel} to their {@link RpcMaturity} counterpart.
     *
     * @param maturity the maturity to convert
     * @return the converted maturity
     * @throws IllegalArgumentException when the given maturity is not mappable
     */
    public static RpcMaturity maturityLevel(final MaturityLevel maturity) {
        final int valueCount = MaturityLevel.values().length;
        if (valueCount > RpcMaturity.values().length) {
            throw new IllegalArgumentException(format(ROOT, "maturity level %s has a too large ordinal: %d", maturity, valueCount));
        }
        return RpcMaturity.values()[maturity.ordinal()];
    }

    /**
     * Convert given {@link PluginType} to their {@link RpcPluginType} counterpart.
     *
     * @param type the type to convert
     * @return the converted type
     * @throws IllegalArgumentException when the given type is not mappable
     */
    public static RpcPluginType pluginType(final PluginType type) {
        final int valueCount = PluginType.values().length;
        if (valueCount > RpcPluginType.values().length) {
            throw new IllegalArgumentException(format(ROOT, "plugin type %s has a too large ordinal: %d", type, valueCount));
        }
        return RpcPluginType.values()[type.ordinal()];
    }

    /**
     * Convert given {@link PluginResources} to their {@link RpcPluginResources} counterpart.
     *
     * @param pluginResources the plugin resources to convert
     * @return the converted plugin resources
     */
    static RpcPluginResources pluginResources(final PluginResources pluginResources) {
        final RpcPluginResources.Builder builder = RpcPluginResources.newBuilder();
        Optional.ofNullable(pluginResources.maximumCpu()).ifPresent(builder::setMaxCpu);
        Optional.ofNullable(pluginResources.maximumMemory()).ifPresent(builder::setMaxMemory);
        Optional.ofNullable(pluginResources.maximumWorkers()).ifPresent(builder::setMaxWorkers);

        return builder.build();
    }

    /**
     * Create an {@link RpcRead} message from given position and size.
     *
     * @param position the position to read at
     * @param count the amount of bytes to read
     * @param traceUid the uid of the trace to read from
     * @param type the type of the data stream to read
     * @return the created message
     */
    public static RpcRead readParameters(final long position, final int count, final String traceUid, final String type) {
        return RpcRead.newBuilder()
            .setPosition(position)
            .setCount(count)
            .setTraceUid(traceUid)
            .setDataType(type)
            .build();
    }

    /**
     * Create a {@link RpcSearchRequest} message from given query and count.
     *
     * @param query string to query
     * @param count maximum number of results returned
     * @param scope scope to limit the search to (image or project)
     * @return the created message
     */
    public static RpcSearchRequest searchRequest(final String query, final int count, final SearchScope scope) {
        return RpcSearchRequest.newBuilder()
            .setCount(count)
            .setQuery(query)
            .setScope(scope == SearchScope.IMAGE ? RpcSearchScope.Image : RpcSearchScope.Project)
            .build();
    }

    /**
     * Packs a (potentially non-homogeneous) List into a RpcList.
     * TODO HANSKEN-21534: Remove non-generic lists and maps in extraction plugins, allowing this method to be removed.
     * @param list The list that we want to pack.
     * @return The packed RpcList.
     */
    public static RpcList genericList(final List<Object> list) {
        final RpcList.Builder builder = RpcList.newBuilder();
        for (final Object value : list) {
            if (value instanceof List) {
                final Any primitive = Any.pack(genericList((List<Object>) value));
                builder.addValues(primitive);
            }
            else if (value instanceof Map) {
                final Any primitive = Any.pack(genericMap((Map<String, Object>) value));
                builder.addValues(primitive);
            }
            else {
                final Any primitive = Any.pack(Pack.primitive(value));
                builder.addValues(primitive);
            }
        }
        return builder.build();
    }

    /**
     * Packs a (potentially non-homogeneous) Map into a RpcMap.
     * TODO HANSKEN-21534: Remove non-generic lists and maps in extraction plugins, allowing this method to be removed.
     * @param map The map that we want to pack.
     * @return The packed RpcMap.
     */
    public static RpcMap genericMap(final Map<String, Object> map) {
        final RpcMap.Builder builder = RpcMap.newBuilder();
        for (final Map.Entry<String, Object> entry : map.entrySet()) {
            final String key = entry.getKey();
            final Object value = entry.getValue();
            if (value instanceof List) {
                final Any primitive = Any.pack(genericList((List<Object>) value));
                builder.putEntries(key, primitive);
            }
            else if (value instanceof Map) {
                final Any primitive = Any.pack(genericMap((Map<String, Object>) value));
                builder.putEntries(key, primitive);
            }
            else {
                final Any primitive = Any.pack(Pack.primitive(value));
                builder.putEntries(key, primitive);
            }
        }
        return builder.build();
    }

    /**
     * Packs an object into a RpcTransformerArgument.
     * @param argument The object that we want to pack as an RpcTransformerArgument.
     * @return The packed RpcTransformerArgument.
     */
    public static RpcTransformerArgument transformerArgument(final Object argument) {
        if (argument instanceof List) {
            return RpcTransformerArgument.newBuilder().setList(genericList((List<Object>) argument)).build();
        }
        else if (argument instanceof Map) {
            return RpcTransformerArgument.newBuilder().setMap(genericMap((Map<String, Object>) argument)).build();
        }
        else {
            final Message message = PRIMITIVE_PACKER.pack(argument);
            final RpcTransformerArgument.Builder builder = RpcTransformerArgument.newBuilder();

            if (message instanceof RpcBoolean) {
                return builder.setBoolean((RpcBoolean) message).build();
            }
            else if (message instanceof RpcBytes) {
                return builder.setBytes((RpcBytes) message).build();
            }
            else if (message instanceof RpcLong) {
                return builder.setInteger((RpcLong) message).build();
            }
            else if (message instanceof RpcDouble) {
                return builder.setReal((RpcDouble) message).build();
            }
            else if (message instanceof RpcString) {
                return builder.setString((RpcString) message).build();
            }
            else if (message instanceof RpcVector) {
                return builder.setVector((RpcVector) message).build();
            }
            else if (message instanceof RpcLatLong) {
                return builder.setLatLong((RpcLatLong) message).build();
            }
            else if (message instanceof RpcZonedDateTime) {
                return builder.setDatetime((RpcZonedDateTime) message).build();
            }
            else {
                throw new IllegalArgumentException("Unsupported message type: " + message.getClass().getName());
            }
        }
    }

    /**
     * Packs a map of TransformerArguments into a RpcTransformerArguments.
     * @param arguments The map of arguments where we want to pack TransformerArguments into RpcTransformerArguments.
     * @return A map where the TransformerArgument values are converted into RpcTransformerArguments.
     */
    public static Map<String, RpcTransformerArgument> transformerArgumentMap(final Map<String, TransformerArgument> arguments) {
        final Map<String, RpcTransformerArgument> map = new HashMap<>();
        for (Map.Entry<String, TransformerArgument> entry : arguments.entrySet()) {
            map.put(entry.getKey(), transformerArgument(entry.getValue().getArgument()));
        }

        return map;
    }

    /**
     * Packs a transformer label describing the transformer and arguments to invoke it into a transformer request.
     * @param label The label containing the name, parameters and return type of the transformer.
     * @param arguments The arguments to invoke the transformer with provided as a (name, value) pair.
     * @return The transformer request that can be sent over grpc to the plugin.
     */
    public static RpcTransformerRequest transformerRequest(final TransformerLabel label, final Map<String, RpcTransformerArgument> arguments) {
        // Construct a RpcTransformerRequest by providing the method name and the arguments.
        // Note: The parameter types are not necessary as the plugin already knows and validates this information.
        return org.hansken.extraction.plugin.grpc.RpcTransformerRequest
            .newBuilder()
            .setMethodName(label.getMethodName())
            .putAllNamedArguments(arguments)
            .build();
    }

    /**
     * Create a {@link RpcEnrichTrace} message which contains given set of types and
     * properties to send to the client and used to enrich the currently processed trace with.
     *
     * @param id the id of the trace to enrich
     * @param types the types to add
     * @param properties the properties to add
     * @param tracelets the tracelets to add
     * @param transformations the transformations to add
     * @return the created message
     */
    public static RpcEnrichTrace traceEnrichment(final String id, final Set<String> types, final Map<String, Object> properties, final List<Trace.Tracelet> tracelets, final Map<String, List<DataTransformation>> transformations) {
        return RpcEnrichTrace.newBuilder()
            .setTrace(Pack.trace(id, types, properties, tracelets, transformations))
            .build();
    }

    /**
     * Create an {@link RpcTrace} from a given set of types and properties, together with
     * the other necessary metadata for a trace.
     *
     * @param id the id of the trace
     * @param types the types of the trace
     * @param properties the properties of the trace
     * @param tracelets the Tracelets to add
     * @param transformations the transformations to add
     * @return the created message
     */
    private static RpcTrace trace(final String id, final Set<String> types, final Map<String, Object> properties, final List<Trace.Tracelet> tracelets, final Map<String, List<DataTransformation>> transformations) {
        final Set<RpcTraceProperty> rpcProperties = properties.keySet().stream()
            .map(property -> Pack.property(property, properties.get(property)))
            .collect(toSet());

        return RpcTrace.newBuilder()
            .setId(id)
            .addAllTypes(types)
            .addAllProperties(rpcProperties)
            .addAllTracelets(rpcTracelets(tracelets))
            .addAllTransformations(dataStreamTransformations(transformations))
            .build();
    }

    /**
     * Create a {@link RpcSearchResult} from a given {@link SearchResult}.
     *
     * @param result the search result containing searched traces
     * @return the created RpcSearchResult
     */
    public static RpcSearchResult searchResult(final SearchResult result) {
        final Iterable<RpcSearchTrace> searchTraces = result.getTraces()
            .map(Pack::searchTrace)
            .collect(toList());

        return RpcSearchResult.newBuilder()
            .setTotalResults(result.getTotalHits())
            .addAllTraces(searchTraces)
            .build();
    }

    /*
     * Create an List of {@link RpcTracelet} from a given set of types and properties, together with
     * the other necessary metadata for a trace.
     *
     * @param allTracelets the Tracelets to add
     * @return the created List of RpcTracelet
     */
    private static List<RpcTracelet> rpcTracelets(final List<Trace.Tracelet> allTracelets) {
        final List<RpcTracelet> rpcAllTracelets = new ArrayList<>();
        allTracelets.forEach(tracelet -> {
            final Set<RpcTraceProperty> rpcTraceProperties = tracelet.getValue().stream()
                .map(traceletProperty -> Pack.property(traceletProperty.getName(), traceletProperty.getValue()))
                .collect(toSet());

            final RpcTracelet rpcTracelet = RpcTracelet.newBuilder()
                .setName(tracelet.getName())
                .addAllProperties(rpcTraceProperties)
                .build();
            rpcAllTracelets.add(rpcTracelet);
        });
        return rpcAllTracelets;
    }

    /**
     * Wrap a {@link Throwable throwable} in a status exception which can be used to signal
     * an error on gRPC stream.
     *
     * @param statusCode the error code
     * @param t the error cause
     * @return a status exception which can be signaled on the gRPC stream
     */
    public static StatusRuntimeException asStatusRuntimeException(final Code statusCode, final Throwable t) {
        return statusCode.toStatus()
            .withDescription(t.getClass().getCanonicalName() + ": " + t.getLocalizedMessage())
            .withCause(t)
            .asRuntimeException();
    }

    static List<RpcDataStreamTransformation> dataStreamTransformations(final Map<String, List<DataTransformation>> dataStreamTransformations) {
        return dataStreamTransformations.entrySet()
            .stream()
            .map(entry -> RpcDataStreamTransformation.newBuilder()
                .setDataType(entry.getKey())
                .addAllTransformations(transformations(entry.getValue()))
                .build())
            .collect(Collectors.toList());
    }

    private static List<RpcTransformation> transformations(final List<DataTransformation> transformations) {
        final List<RpcTransformation> rpcDataTransformations = new ArrayList<>();
        for (final DataTransformation transformation : transformations) {
            rpcDataTransformations.add(getRpcDataTransformation((RangedDataTransformation) transformation));
        }
        return rpcDataTransformations;
    }

    private static RpcTransformation getRpcDataTransformation(final RangedDataTransformation rangedTransformation) {
        final List<RpcRange> ranges = rangedTransformation.getRanges()
            .stream()
            .map(Pack::getRpcRange)
            .collect(Collectors.toList());
        final RpcRangedTransformation rpcRangedDataTransformation = RpcRangedTransformation.newBuilder()
            .addAllRanges(ranges)
            .build();
        return RpcTransformation.newBuilder()
            .setRangedTransformation(rpcRangedDataTransformation)
            .build();
    }

    private static RpcRange getRpcRange(final DataRange dataRange) {
        return RpcRange.newBuilder()
            .setLength(dataRange.getLength())
            .setOffset(dataRange.getOffset())
            .build();
    }

    static final class Packer {

        private final Map<Class<?>, Function<Object, Message>> _packers = new HashMap<>();

        static Packer create() {
            return new Packer();
        }

        <T> Packer register(final Class<T> objectType, final Function<T, Message> packer) {
            _packers.put(objectType, (Function<Object, Message>) packer);
            return this;
        }

        Message pack(final Object object) {
            if (object == null) {
                return NULL;
            }
            for (final var packer : _packers.entrySet()) {
                if (packer.getKey().isInstance(object)) {
                    return packer.getValue().apply(object);
                }
            }
            throw new IllegalArgumentException("unhandled type of object: " + object.getClass());
        }
    }
}
