package org.hansken.plugin.extraction.hql_lite.lang.human;

import static org.hansken.plugin.extraction.hql_lite.lang.human.HqlHumanUtil.unescape;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.all;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.and;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.data;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.dataType;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.not;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.or;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.range;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.term;
import static org.hansken.plugin.extraction.hql_lite.lang.human.matcher.QueryBuilders.type;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import org.antlr.v4.runtime.ANTLRErrorListener;
import org.antlr.v4.runtime.CharStreams;
import org.antlr.v4.runtime.CommonTokenStream;
import org.apache.commons.lang3.math.NumberUtils;
import org.hansken.plugin.extraction.api.Trace;
import org.hansken.plugin.extraction.hql_lite.lang.human.matcher.HqlMatcher;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanLexer;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanParser;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanParser.ComplexExprContext;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanParser.DataTypeExprContext;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanParser.FieldsContext;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanParser.QueryContext;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanParser.StringValueContext;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanParser.TermExprContext;
import org.hansken.plugin.extraction.hql_lite.lang.parsers.hqlhuman.HqlHumanParser.TypeExprContext;

/**
 * Parser implementation for Hql-lite queries.
 *
 * @author Netherlands Forensic Institute
 */
public final class HqlLiteHumanQueryParser {
    private static final String[] TEXT_FIELDS = new String[]{"text"}; // TODO remove in HANSKEN-16009
    private static final String[] META_FIELDS = new String[]{"meta"};

    private HqlLiteHumanQueryParser() {
    }

    /**
     * This method parses an hql-lite query and creates a {@link HqlMatcher} matcher that can be used to verify whether
     * an incoming {@link Trace} can be processed by the external plugin.
     *
     * @param query an hql-lite query
     * @return a {@link HqlMatcher} matcher instance
     */
    public static HqlMatcher parse(final String query) {
        final HqlHumanParser parser = getParser(query);
        final QueryContext context = parser.query();
        final HqlMatcher matcherObj = visit(context);
        return matcherObj;
    }

    private static HqlHumanParser getParser(final String query) {
        final ANTLRErrorListener listener = new ParserErrorListener(HqlLiteHumanQueryParser.class);
        final HqlHumanLexer lexer = new HqlHumanLexer(CharStreams.fromString(query));
        lexer.removeErrorListeners();
        lexer.addErrorListener(listener);
        final CommonTokenStream tokens = new CommonTokenStream(lexer);
        final HqlHumanParser parser = new HqlHumanParser(tokens);
        parser.removeErrorListeners();
        parser.addErrorListener(listener);
        return parser;
    }

    private static HqlMatcher visit(final QueryContext ctx) {
        return visit(ctx.complexExpr());
    }

    private static HqlMatcher visit(final ComplexExprContext ctx) {
        if (ctx == null) {
            return all();
        }
        else if (ctx.and() != null) {
            return and(visit(ctx.complexExpr(0)), visit(ctx.complexExpr(1)));
        }
        else if (ctx.or() != null) {
            return or(visit(ctx.complexExpr(0)), visit(ctx.complexExpr(1)));
        }
        else if (ctx.not() != null) {
            return not(visit(ctx.complexExpr(0)));
        }
        else if (ctx.termExpr() != null) {
            return visit(ctx.termExpr());
        }
        else if (ctx.rangeExpr() != null) {
            return visit(ctx.rangeExpr());
        }
        else if (ctx.regexExpr() != null) {
            return visit(ctx.regexExpr());
        }
        else if (ctx.phraseExpr() != null) {
            return visit(ctx.phraseExpr());
        }
        else if (ctx.hasTraceExpr() != null) {
            return visit(ctx.hasTraceExpr());
        }
        else if (ctx.hasTraceletExpr() != null) {
            return visit(ctx.hasTraceletExpr());
        }
        else if (ctx.nestedExpr() != null) {
            return visit(ctx.nestedExpr());
        }
        else if (ctx.dataTypeExpr() != null) {
            return visit(ctx.dataTypeExpr());
        }
        else if (ctx.typeExpr() != null) {
            return visit(ctx.typeExpr());
        }
        else {
            final List<ComplexExprContext> complexExprs = ctx.complexExpr();
            switch (complexExprs.size()) {
                case 1:
                    return visit(complexExprs.get(0));
                case 2:
                    return and(visit(complexExprs.get(0)), visit(complexExprs.get(1)));
                default:
                    throw new IllegalStateException("unexpected number of complex exprs: " + complexExprs.size());
            }
        }
    }

    private static HqlMatcher visit(final TermExprContext ctx) {
        // fullmatch term-queries are not supported on data, so when no field is given, only meta-data is searched
        // (if the default text field were used, all queries would return an error: "fullmatch termquery not supported on data")
        final boolean fullMatch = isFullMatch(ctx.stringValue());
        final boolean isDataStream = isDataStream(ctx.dataFields());
        final FieldsContext fieldsCtx = isDataStream ? ctx.dataFields().fields() : ctx.fields();
        final String[] fields = fields(fieldsCtx, fullMatch);
        final StringValueContext valueContext = ctx.stringValue();
        final String value = HqlHumanUtil.unescapeKeepWildcards(visit(valueContext));
        final HqlMatcher query = term(value, fullMatch, fields);

        final HqlMatcher hqlMatcher = ctx.NEQ() == null ? query : not(query);
        return isDataStream ? data(hqlMatcher) : hqlMatcher;
    }

    private static HqlMatcher visit(final DataTypeExprContext ctx) {
        final StringValueContext stringValue = ctx.stringValue();
        final boolean fullMatch = isFullMatch(stringValue);
        final String value = HqlHumanUtil.unescapeKeepWildcards(visit(stringValue));
        final HqlMatcher query = dataType(value, fullMatch);

        return ctx.NEQ() == null ? query : not(query);
    }

    private static HqlMatcher visit(final TypeExprContext ctx) {
        final StringValueContext stringValue = ctx.stringValue();
        final List<StringValueContext> stringValueContexts = flattenBinaryTreeList(stringValue);

        final List<HqlMatcher> subQueries = new ArrayList<>(stringValueContexts.size());
        for (final StringValueContext valueContext : stringValueContexts) {
            final boolean fullMatch = isFullMatch(valueContext);
            final String value = HqlHumanUtil.unescapeKeepWildcards(visit(valueContext));
            final HqlMatcher term = term(value, fullMatch, "type");
            subQueries.add(term);
        }

        final HqlMatcher query = type(subQueries);
        return ctx.NEQ() == null ? query : not(query);
    }

    private static List<StringValueContext> flattenBinaryTreeList(final StringValueContext stringValue) {
        final List<StringValueContext> theList = new ArrayList<>();
        if (stringValue.stringValue().isEmpty()) {
            theList.add(stringValue);
        }
        else {
            // comma separated strings are parsed into a binary tree collection of strings
            // example:
            //    A,B,C
            //     /\
            //    A  B,C
            //       /\
            //      B  C
            //
            for (final StringValueContext stringValueContext : stringValue.stringValue()) {
                theList.addAll(flattenBinaryTreeList(stringValueContext));
            }
        }
        return theList;
    }

    private static boolean isFullMatch(final StringValueContext stringValue) {
        if (stringValue != null) {
            // single quoted strings indicate full-match term-queries
            final String text = stringValue.getText();
            return text != null && text.length() > 0 && text.charAt(0) == '\'';
        }
        return false;
    }

    private static String visit(final StringValueContext ctx) {
        return HqlStringUtil.unquoteSingle(ctx.getText());
    }

    private static String[] fields(final FieldsContext ctx, final boolean fullMatch) {
        return fields(ctx, fullMatch ? META_FIELDS : TEXT_FIELDS);
    }

    private static String[] fields(final FieldsContext ctx, final String[] missing) {
        return ctx == null ? missing : visit(ctx);
    }

    private static String[] visit(final FieldsContext ctx) {
        final String[] fields = new String[ctx.field().size()];
        for (int i = 0; i < fields.length; i++) {
            fields[i] = unescape(HqlStringUtil.unquoteSingle(ctx.field(i).getText()));
        }
        return fields;
    }

    private static HqlMatcher visit(final HqlHumanParser.RangeExprContext ctx) {
        final boolean fullMatch = isFullMatch(ctx.value(0).stringValue());
        final boolean isDataStream = isDataStream(ctx.dataFields());
        final FieldsContext fieldsCtx = isDataStream ? ctx.dataFields().fields() : ctx.fields();
        final String[] fields = fields(fieldsCtx, fullMatch);
        final HqlMatcher query;

        if (ctx.TO() != null) {
            // field: min..max
            query = range(value(ctx.value(0)), value(ctx.value(1)), true, fullMatch, fields);
        }
        else if (ctx.value().size() == 2) {
            // min </= field </= max
            final BigDecimal[] values = new BigDecimal[2];
            values[0] = value(ctx.value(0));
            values[1] = value(ctx.value(1));

            final boolean includesMin = ctx.SMALLER(0).getText().length() == 2; //Length is either 1 or 2 (> vs >=)
            final boolean includesMax = ctx.SMALLER(1).getText().length() == 2;

            query = range(values[0], values[1], includesMin, includesMax, fullMatch, fields);
        }
        else {
            // field </=/> value
            final BigDecimal value = value(ctx.value(0));
            final String op = ctx.GREATER() != null ? ctx.GREATER().getText() : ctx.SMALLER(0).getText();

            final boolean includesMinMax = op.length() == 2;
            if (op.charAt(0) == '>') {
                query = range(value, null, includesMinMax, false, fullMatch, fields);
            }
            else {
                query = range(null, value, includesMinMax, fullMatch, fields);
            }
        }

        return isDataStream ? data(query) : query;
    }

    private static boolean isDataStream(final HqlHumanParser.DataFieldsContext fields) {
        return fields != null;
    }

    private static HqlMatcher visit(final HqlHumanParser.RegexExprContext ctx) {
        throw new UnsupportedOperationException("Regex queries are not implemented yet.");
    }

    private static HqlMatcher visit(final HqlHumanParser.PhraseExprContext ctx) {
        throw new UnsupportedOperationException("Phrase queries are not implemented yet.");
    }

    private static HqlMatcher visit(final HqlHumanParser.HasTraceExprContext ctx) {
        throw new UnsupportedOperationException("Trace queries are not implemented yet.");
    }

    private static HqlMatcher visit(final HqlHumanParser.HasTraceletExprContext ctx) {
        throw new UnsupportedOperationException("Tracelet queries are not implemented yet.");
    }

    private static HqlMatcher visit(final HqlHumanParser.NestedExprContext ctx) {
        throw new UnsupportedOperationException("Nested queries are not implemented yet.");
    }

    private static BigDecimal value(final HqlHumanParser.ValueContext ctx) {
        if (ctx.latlongValue() != null) {
            throw new UnsupportedOperationException("RangeQuery: LatLong range not implemented yet.");
        }
        if (ctx.stringValue() != null) {
            final String unescapedString = unescape(visit(ctx.stringValue()));
            if (NumberUtils.isCreatable(unescapedString)) {
                return new BigDecimal(unescapedString);
            }
            throw new UnsupportedOperationException("RangeQuery: String range not implemented yet.");
        }

        throw new IllegalStateException("no such value context: " + ctx);
    }
}
