package org.hansken.plugin.extraction.hql_lite.lang.human.matcher;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotAllNull;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotEmpty;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;
import static org.hansken.plugin.extraction.util.ArgChecks.argsIsType;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import org.hansken.plugin.extraction.api.Trace;

/**
 * Query builders.
 *
 * @author Netherlands Forensic Institute
 */
public final class QueryBuilders {
    private static final AllHqlMatcher ALL_QUERY = new AllHqlMatcher();
    private static final NoneHqlMatcher NONE_QUERY = new NoneHqlMatcher();

    private QueryBuilders() {
    }

    /**
     * Matches all traces.
     *
     * @return AllQuery
     */
    public static AllHqlMatcher all() {
        return ALL_QUERY;
    }

    /**
     * Builds a TermQuery, matching all traces that have any of the supplied properties equal to the supplied value(s).
     *
     * @param value the value against which the set propertie(s) should be matched. a string with optional wildcards(? *).
     * @param fullMatch whether to match full(treating wildcards as string literals), untokenized string values, default false.
     * @param properties the properties to match.
     * @return TermQuery
     */
    public static TermHqlMatcher term(final String value, final boolean fullMatch, final String... properties) {
        argNotNull("value", value);
        argNotEmpty("properties", properties);
        return new TermHqlMatcher(properties[0], fullMatch, value);
    }

    /**
     * Overloaded version of {@link QueryBuilders#term(String, boolean, String...)}, with default value
     * {@code fullMatch = false}.
     *
     * @param value the value against which the set propertie(s) should be matched. a string with optional wildcards(? *).
     * @param properties the properties to match.
     * @return TermQuery
     */
    public static TermHqlMatcher term(final String value, final String... properties) {
        return term(value, false, properties);
    }

    /**
     * Builds a {@link DataHqlMatcher}, matching the {@code matcher} with the HQL query expanding with the current
     * datastream.
     *
     * @param matcher the properties to match.
     * @return DataHqlMatcher
     */
    public static DataHqlMatcher data(final HqlMatcher matcher) {
        argNotNull("matcher", matcher);
        return new DataHqlMatcher(matcher);
    }

    /**
     * Builds a DataHqlMatcher, matching all traces where the dataType is equal to the supplied value(s).
     *
     * @param value the value against which the set propertie(s) should be matched. a string with optional wildcards(? *).
     * @param fullMatch whether to match full(treating wildcards as string literals), untokenized string values, default false.
     * @return DataTypeHqlMatcher
     */
    public static DataTypeHqlMatcher dataType(final String value, final boolean fullMatch) {
        argNotEmpty("value", value);
        return new DataTypeHqlMatcher(value, fullMatch);
    }

    /**
     * Overloaded version of {@link QueryBuilders#dataType(String, boolean)}, with
     * default value {@code fullMatch = false}.
     *
     * @param value the value against which the set propertie(s) should be matched. a string with optional wildcards(? *).
     * @return DataTypeHqlMatcher
     */
    public static DataTypeHqlMatcher dataType(final String value) {
        return dataType(value, false);
    }

    /**
     * Builds a TypeHqlMatcher, matching all traces that have <b>all</b> of the supplied {@link Trace#types()} equal
     * to the values supplied with {@code matchers}.
     *
     * @param matchers a list of matchers for every type we want to match on
     * @return TypesHqlMatcher
     */
    public static TypeHqlMatcher type(final List<HqlMatcher> matchers) {
        argNotEmpty("matchers", matchers);
        argsIsType("matchers", matchers, TermHqlMatcher.class);

        return new TypeHqlMatcher(new ArrayList(matchers));
    }

    /**
     * Builds a RangeQuery, matching all traces that have any of the supplied properties within a range of values.
     *
     * @param min minimum value of the range
     * @param max maximum value of the range
     * @param includesMin is the min value inclusive(greater than <b>or equal</b>). default true.
     * @param includesMax is the max value inclusive(less than or <b>equal</b>)
     * @param fullMatch currently does nothing
     * @param properties the properties to match.
     * @return RangeQuery
     */
    public static NumberRangeHqlMatcher range(final BigDecimal min, final BigDecimal max, final boolean includesMin, final boolean includesMax, final boolean fullMatch, final String... properties) {
        argNotAllNull("min, max", min, max);
        argNotEmpty("properties", properties);
        return new NumberRangeHqlMatcher(min, max, includesMin, includesMax, fullMatch, properties[0]);
    }

    /**
     * Overloaded version {@link QueryBuilders#range(BigDecimal, BigDecimal, boolean, boolean, boolean, String...)} with
     * default value {@code includesMin = false}.
     *
     * @param min minimum value of the range
     * @param max maximum value of the range
     * @param includesMax is the max value inclusive(less than or <b>equal</b>)
     * @param fullMatch currently does nothing
     * @param properties the properties to match.
     * @return RangeQuery
     */
    public static NumberRangeHqlMatcher range(final BigDecimal min, final BigDecimal max, final boolean includesMax, final boolean fullMatch, final String... properties) {
        return range(min, max, true, includesMax, fullMatch, properties);
    }

    /**
     * Overloaded version {@link QueryBuilders#range(BigDecimal, BigDecimal, boolean, boolean, boolean, String...)} that
     * converts {@code min} and/or {@code max} from {@link String} to {@link BigDecimal}.
     *
     * @param min minimum value of the range
     * @param max maximum value of the range
     * @param includesMin is the min value inclusive(greater than <b>or equal</b>). default true.
     * @param includesMax is the max value inclusive(less than or <b>equal</b>)
     * @param fullMatch currently does nothing
     * @param properties the properties to match.
     * @return RangeQuery
     */
    public static NumberRangeHqlMatcher range(final String min, final String max, final boolean includesMin, final boolean includesMax, final boolean fullMatch, final String... properties) {
        return range(toBigDecimal(min), toBigDecimal(max), includesMin, includesMax, fullMatch, properties);
    }

    /**
     * Overloaded version {@link QueryBuilders#range(String, String, boolean, boolean, boolean, String...)} with
     * default value {@code includesMin = false}.
     *
     * @param min minimum value of the range
     * @param max maximum value of the range
     * @param includesMax is the max value inclusive(less than or <b>equal</b>)
     * @param fullMatch currently does nothing
     * @param properties the properties to match.
     * @return RangeQuery
     */
    public static NumberRangeHqlMatcher range(final String min, final String max, final boolean includesMax, final boolean fullMatch, final String... properties) {
        return range(min, max, true, includesMax, fullMatch, properties);
    }

    /**
     * Builds an AndQuery, matching all traces that match all subqueries.
     *
     * @param queries the queries to match.
     * @return AndQuery
     */
    public static AndHqlMatcher and(final HqlMatcher... queries) {
        argNotNull("queries", queries);
        return new AndHqlMatcher(queries);
    }

    /**
     * Builds an OrQuery, matching all traces that match any subquery.
     *
     * @param queries the queries to match.
     * @return OrQuery
     */
    public static OrHqlMatcher or(final HqlMatcher... queries) {
        argNotNull("queries", queries);
        return new OrHqlMatcher(queries);
    }

    /**
     * Builds an NotQuery, matching all traces that do not match the subquery.
     *
     * @param matcher the query not to match.
     * @return NotQuery
     */
    public static NotHqlMatcher not(final HqlMatcher matcher) {
        argNotNull("query", matcher);
        return new NotHqlMatcher(matcher);
    }

    private static BigDecimal toBigDecimal(final String min) {
        if (min == null) {
            return null;
        }
        return new BigDecimal(min);
    }
}
