package org.hansken.plugin.extraction.hql_lite.lang.human.matcher;

import static org.hansken.plugin.extraction.api.PluginType.DEFERRED_EXTRACTION_PLUGIN;
import static org.hansken.plugin.extraction.api.PluginType.EXTRACTION_PLUGIN;
import static org.hansken.plugin.extraction.api.PluginType.META_EXTRACTION_PLUGIN;

import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.ImmutableTrace;
import org.hansken.plugin.extraction.api.PluginType;
import org.hansken.plugin.extraction.api.Trace;

/**
 * This interface specifies the minimum required methods that a compiled Matcher node needs to run.
 * <p>
 * Before the extraction process, Hansken will query the Extraction plugin for the PluginInfo object,
 * which also exposes the HQL string provided by the plugin, which is then compiled to a {@link HqlMatcher} instance.
 * This way, the extraction process can match the traces which
 * should be sent to the ExtractionPlugin for processing, to avoid sending irrelevant data to the plugin.
 * Delegating the matching process to the ExtractionPlugin itself would cause overhead, because in that case,
 * every trace should be sent to the plugin in order to determine if it should be processed.
 */
public interface HqlMatcher {

    /**
     * This method matches the conditions defined in the query against the trace input.
     *
     * @param trace    the trace to process
     * @param dataType the type of the current trace processing iteration
     * @return true if the input trace satisfies the conditions of the query
     */
    boolean match(ImmutableTrace trace, String dataType);

    /**
     * Getter for the property we would like to match.
     *
     * @return the property we would like to match against.
     */
    default String property() {
        throw new UnsupportedOperationException();
    }

    /**
     * Getter for the match full flag.
     *
     * @return the value of the flag of whether to match full(treating wildcards as string literals), untokenized string values, default false.
     */
    default boolean fullMatch() {
        return false;
    }

    /**
     * Check that given {@link HqlMatcher matcher} matches given {@link Trace trace} and {@code dataType},
     * taking into account which type of {@link ExtractionPlugin plugin} we are dealing with.
     * <p>
     * Currently, this means that matching on a {@code 'meta'} data type can only succeed when we have a plugin
     * of type {@link PluginType#META_EXTRACTION_PLUGIN}. In all other cases, it can succeed for a plugin
     * of type {@link PluginType#EXTRACTION_PLUGIN} or {@link PluginType#DEFERRED_EXTRACTION_PLUGIN}. Note that
     * even then, matching still only succeeds when the matcher does.
     * <p>
     * <strong>Note:</strong> this function does no parameter validation, so care must be taken by the client.
     *
     * @param type     the type of the plugin
     * @param matcher  the matcher to use
     * @param trace    the trace to match on
     * @param dataType the data type contained in the current context
     * @return {@code true} if the match succeeded, otherwise {@code false}
     */
    static boolean matches(final PluginType type, final HqlMatcher matcher, final Trace trace, final String dataType) {
        if ("meta".equals(dataType)) {
            return type == META_EXTRACTION_PLUGIN && matcher.match(trace, dataType);
        }

        return (type == EXTRACTION_PLUGIN || type == DEFERRED_EXTRACTION_PLUGIN) && matcher.match(trace, dataType);
    }
}
