package org.hansken.plugin.extraction.hql_lite.lang.human.matcher;

import static org.hansken.plugin.extraction.hql_lite.lang.human.HqlHumanUtil.swapEscapedWildcards;
import static org.hansken.plugin.extraction.hql_lite.lang.human.HqlHumanUtil.unswapEscapedWildcards;

import java.util.Collection;

final class HqlMatcherUtil {
    private static final RegexCache REGEX_CACHE = new RegexCache();

    private HqlMatcherUtil() {
    }

    /**
     * This function does a wildcard based comparison of the incoming {@code traceValue} against the {@link HqlMatcher}
     * {@code matcherValue} pattern.
     *
     * @param matcherValue the value we are comparing to
     * @param traceValue the value of the incoming trace
     * @return true if a match is found
     */
    static boolean matchStringValuesWithWildcards(final String matcherValue, final String traceValue) {
        if (traceValue == null) {
            return false;
        }

        final String regexValue = wildcardToRegex(matcherValue);
        return REGEX_CACHE.compileRegex("^" + regexValue + "$").test(traceValue);
    }

    /**
     * This function does a wildcard based comparison of the incoming {@code traceValues} against the {@link HqlMatcher}
     * {@code matcherValue} pattern. In other words, it's a {@link Collection#contains(java.lang.Object)} but with
     * wildcard support.
     *
     * @param matcherValue the value we are comparing to
     * @param traceValues the values of the incoming trace
     * @return true if a match is found
     */
    static boolean matchStringValuesWithWildcards(final String matcherValue, final Collection<String> traceValues) {
        if (matcherValue == null) {
            return false;
        }

        for (String traceValue : traceValues) {
            if (matchStringValuesWithWildcards(matcherValue, traceValue)) {
                return true;
            }
        }
        return false;
    }

    private static String wildcardToRegex(final String str) {
        final String encoded = swapEscapedWildcards(str);

        final String regex =
            encoded // replace (unescaped) wildcards with regex equivalents
                .replace("?", ".")
                .replace("*", ".*");

        return unswapEscapedWildcards(regex);
    }
}
