package org.hansken.plugin.extraction.hql_lite.lang.human.matcher;

import java.math.BigDecimal;
import java.util.Objects;

import org.hansken.plugin.extraction.api.ImmutableTrace;

/**
 * This matcher checks whether a value is within a specific range. both inclusive and/or exclusive ranges are supported.
 */
class NumberRangeHqlMatcher implements HqlMatcher, DataStreamMatcher {
    private final BigDecimal _min;
    private final BigDecimal _max;
    private final boolean _includesMin;
    private final boolean _includesMax;
    private final boolean _fullMatch;
    private final String _property;

    NumberRangeHqlMatcher(final BigDecimal min, final BigDecimal max, final boolean includesMax, final boolean fullMatch, final String property) {
        _min = min;
        _max = max;
        _includesMin = true;
        _includesMax = includesMax;
        _fullMatch = fullMatch;
        _property = property;
    }

    NumberRangeHqlMatcher(final BigDecimal min, final BigDecimal max, final boolean includesMin, final boolean includesMax, final boolean fullMatch, final String property) {
        _min = min;
        _max = max;
        _includesMin = includesMin;
        _includesMax = includesMax;
        _fullMatch = fullMatch;
        _property = property;
    }

    @Override
    public boolean match(final ImmutableTrace trace, final String dataType) {
        return matchDataStream(trace, _property);
    }

    @Override
    public boolean matchDataStream(final ImmutableTrace trace, final String traceDataProperty) {
        if (!trace.properties().contains(traceDataProperty)) {
            return false;
        }
        final BigDecimal value = new BigDecimal(Objects.toString(trace.get(traceDataProperty)));

        boolean result = true;
        if (_min != null) {
            if (_includesMin) {
                result &= value.compareTo(_min) >= 0;
            }
            else {
                result &= value.compareTo(_min) > 0;
            }
        }
        if (_max != null) {
            if (_includesMax) {
                result &= value.compareTo(_max) <= 0;
            }
            else {
                result &= value.compareTo(_max) < 0;
            }
        }
        return result;
    }

    @Override
    public String property() {
        return _property;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final NumberRangeHqlMatcher that = (NumberRangeHqlMatcher) o;
        return _includesMin == that._includesMin &&
            _includesMax == that._includesMax &&
            _fullMatch == that._fullMatch &&
            Objects.equals(_min, that._min) &&
            Objects.equals(_max, that._max) &&
            Objects.equals(_property, that._property);
    }

    @Override
    public int hashCode() {
        return Objects.hash(_min, _max, _includesMin, _includesMax, _fullMatch, _property);
    }
}
