package org.hansken.plugin.extraction.hql_lite.lang.human.matcher;

import java.time.Duration;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

/**
 * A simple <b>thread-safe</b> cache for regular expressions.
 */
final class RegexCache {
    private static final Duration DEFAULT_EXPIRE_DURATION = Duration.ofDays(1);
    private static final long DEFAULT_MAX_SIZE = 1000;

    private final Cache<String, Pattern> _cache;

    RegexCache() {
        _cache = Caffeine.newBuilder()
            .expireAfterWrite(DEFAULT_EXPIRE_DURATION)
            .maximumSize(DEFAULT_MAX_SIZE)
            .build();
    }

    /**
     * Compiles and caches the provided {@code regexString}. The {@code regexString} is ONLY compiled if it is not
     * already present in the cache.
     *
     * @param regexString a valid regular expression
     * @return the cached {@link Pattern#asMatchPredicate()} representation of the regex
     */
    Predicate<String> compileRegex(final String regexString) {
        // To let ".*" match on multiline strings as well we have to pass the DOTALL flag, because the dot does not
        // match on newlines by default
        return _cache.get(regexString, string -> Pattern.compile(string, Pattern.DOTALL)).asMatchPredicate();
    }
}
