package org.hansken.plugin.extraction.hql_lite.lang.human.matcher;

import java.util.List;
import java.util.Objects;

import org.hansken.plugin.extraction.api.ImmutableTrace;

/**
 * Either fully or partially matches a property/value kv pair with certain values and wildcards.<br>
 * The property serves as a key that is looked up in the provided trace that results in a traceValue that is compared AGAINST a user provided _value.
 */
class TermHqlMatcher implements DataStreamMatcher, HqlMatcher {
    private final String _property;
    private final boolean _fullMatch;
    private final String _value;

    TermHqlMatcher(final String property, final boolean fullMatch, final String value) {
        _property = property;
        _fullMatch = fullMatch;
        _value = value;
    }

    @Override
    public boolean match(final ImmutableTrace trace, final String dataType) {
        return matchDataStream(trace, _property);
    }

    @Override
    public boolean matchDataStream(final ImmutableTrace trace, final String traceDataProperty) {
        final Object traceValue = trace.get(traceDataProperty);
        return matchOnType(traceValue);
    }

    private boolean matchOnType(final Object traceValue) {
        if (traceValue == null) {
            return false;
        }
        if (traceValue instanceof String) {
            return HqlMatcherUtil.matchStringValuesWithWildcards(_value, (String) traceValue);
        }
        if (traceValue instanceof Long) {
            return HqlMatcherUtil.matchStringValuesWithWildcards(_value, traceValue.toString());
        }
        if (traceValue instanceof List<?>) {
            for (final Object element : ((List<?>) traceValue)) {
                if (matchOnType(element)) {
                    return true;
                }
            }
            return false;
        }
        throw new UnsupportedOperationException("Unsupported type:" + traceValue.getClass());
    }

    @Override
    public final String property() {
        return _property;
    }

    public String value() {
        return _value;
    }

    @Override
    public boolean fullMatch() {
        return _fullMatch;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final TermHqlMatcher termQuery = (TermHqlMatcher) o;
        return _fullMatch == termQuery._fullMatch &&
            Objects.equals(_property, termQuery._property) &&
            Objects.equals(_value, termQuery._value);
    }

    @Override
    public int hashCode() {
        return Objects.hash(_property, _fullMatch, _value);
    }
}
