package org.hansken.plugin.extraction.runtime.grpc.server;

import java.io.IOException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.hansken.plugin.extraction.api.BaseExtractionPlugin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.grpc.BindableService;
import io.grpc.Server;
import io.grpc.ServerBuilder;

/**
 * gRPC Server for Extraction Plugins.
 *
 * @author Netherlands Forensic Institute
 */
public final class ExtractionPluginServer implements AutoCloseable {
    /**
     * Number of simultaneous workers.
     */
    public static final int DEFAULT_NUM_WORKERS = 12;

    // TODO: (MVP) tune variables
    private static final int MAX_MESSAGE_SIZE = 64 * 1024 * 1024;
    private static final int SHUTDOWN_TIMEOUT_SECONDS = 30;
    private static final Logger LOG = LoggerFactory.getLogger(ExtractionPluginServer.class);

    private final ExecutorService _workers;
    private volatile Server _grpcServer;

    private ExtractionPluginServer(final int port, final int nThreads, final BindableService service) {
        _workers = Executors.newFixedThreadPool(nThreads);
        _grpcServer = ServerBuilder
            .forPort(port)
            .addService(new HealthService())
            .addService(service)
            .maxInboundMessageSize(MAX_MESSAGE_SIZE)
            .executor(_workers)
            .build();
    }

    /**
     * Returns the local port the rpc-server is listening on.
     *
     * @return The local port the rpc-server is listening on.
     */
    public int getListeningPort() {
        return _grpcServer.getPort();
    }

    /**
     * Starts a Extraction Plugin server, and returns the server instance. This method starts the
     * server in a separate thread.
     * <br>
     * Use this method to start a new extraction plugin server instance, e.g.
     *   <code>serve(8111, SimplePlugin::new);</code>
     *
     * @param port port on which the extraction plugin is served
     * @param plugin supplier that provides a new instance of the plugin
     * @return a GRPC server instance serving the plugin
     * @throws Exception if the plugin can not be served
     * @deprecated use {@link ExtractionPluginServer#serve(int, int, java.util.function.Supplier)} instead
     */
    @Deprecated
    public static ExtractionPluginServer serve(final int port, final Supplier<BaseExtractionPlugin> plugin) throws Exception {
        return serve(port, DEFAULT_NUM_WORKERS, plugin);
    }

    /**
     * Starts a Extraction Plugin server, and returns the server instance. This method starts the
     * server in a separate thread.
     * <br>
     * Use this method to start a new extraction plugin server instance, e.g.
     *   <code>serve(8111, SimplePlugin::new);</code>
     *
     * @param port port on which the extraction plugin is served
     * @param numberOfWorkers the size of the underlying threadpool, which is equal to the number of concurrent workers
     * @param plugin supplier that provides a new instance of the plugin
     * @return a GRPC server instance serving the plugin
     * @throws Exception if the plugin can not be served
     */
    public static ExtractionPluginServer serve(final int port, final int numberOfWorkers, final Supplier<BaseExtractionPlugin> plugin) throws Exception {
        LOG.info("Starting Hansken extraction plugin on port " + port + " with " + numberOfWorkers + " workers");
        return serve(new ExtractionPluginServer(port, numberOfWorkers, new ExtractionPluginServerService(plugin, MAX_MESSAGE_SIZE)));
    }

    /**
     * Start a server with a custom service implementation.
     * <p>
     * <strong>Note: </strong> this can be used to inject a custom service. This is
     * generally only useful for testing, so that custom service behaviour can be mocked.
     *
     * @param port on which the service is to be served
     * @param service the service to server
     * @return a server instance
     * @throws Exception if the service cannot be served
     * @deprecated use {@link ExtractionPluginServer#serve(int, int, io.grpc.BindableService)} instead
     */
    @Deprecated
    public static ExtractionPluginServer serve(final int port, final BindableService service) throws Exception {
        return serve(port, DEFAULT_NUM_WORKERS, service);
    }

    /**
     * Start a server with a custom service implementation.
     * <p>
     * <strong>Note: </strong> this can be used to inject a custom service. This is
     * generally only useful for testing, so that custom service behaviour can be mocked.
     *
     * @param port on which the service is to be served
     * @param numberOfWorkers the size of the underlying threadpool, which is equal to the number of concurrent workers
     * @param service the service to server
     * @return a server instance
     * @throws Exception if the service cannot be served
     */
    public static ExtractionPluginServer serve(final int port, final int numberOfWorkers, final BindableService service) throws Exception {
        LOG.info("Starting Hansken extraction plugin on port " + port);
        return serve(new ExtractionPluginServer(port, numberOfWorkers, service));
    }

    public static ExtractionPluginServer serve(final ExtractionPluginServer server) throws Exception {
        final CountDownLatch latch = new CountDownLatch(1);
        final Thread thread = new Thread(() -> {
            try {
                server.start();
                latch.countDown();
                server.blockUntilShutdown();
            }
            catch (final InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new IllegalStateException(e);
            }
            catch (final Exception e) {
                throw new IllegalStateException(e);
            }
        });
        thread.start();
        latch.await();
        final int port = server.getListeningPort();
        thread.setName("grpc-extractionplugin-server@" + port);
        LOG.info("Hansken extraction plugin listening port " + port);

        return server;
    }

    /**
     * Stop serving requests and shutdown resources.
     */
    @Override
    public void close() {
        if (_grpcServer != null) {
            try {
                _grpcServer.shutdownNow().awaitTermination(SHUTDOWN_TIMEOUT_SECONDS, TimeUnit.SECONDS);
            }
            catch (final InterruptedException e) {
                Thread.currentThread().interrupt();
                LOG.error("Server shutdown timeout", e);
            }
            _grpcServer = null;
        }
    }

    private void start() throws IOException {
        _grpcServer.start();
    }

    void blockUntilShutdown() throws InterruptedException {
        if (_grpcServer != null) {
            _grpcServer.awaitTermination();
        }
    }

    // default visibility for access in test
    ExecutorService workers() {
        return _workers;
    }
}
