package org.hansken.plugin.extraction.runtime.grpc.server.logging;

import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.appender.ConsoleAppender;
import org.apache.logging.log4j.core.config.Configuration;
import org.apache.logging.log4j.core.config.ConfigurationFactory;
import org.apache.logging.log4j.core.config.ConfigurationSource;
import org.apache.logging.log4j.core.config.builder.api.ConfigurationBuilder;
import org.apache.logging.log4j.core.config.builder.impl.BuiltConfiguration;
import org.apache.logging.log4j.util.LoaderUtil;
import org.apache.logging.log4j.util.Strings;

/**
 * Returns default Configuration if no logging configuration file is found.
 */
abstract class BaseConfigurationFactory extends ConfigurationFactory {
    private static final String ALL_TYPES = "*";

    /**
     * Returns the Configuration.
     *
     * @param loggerContext  The logger context
     * @param name           The configuration name
     * @param configLocation The initial configuration location or null
     * @return The Configuration
     */
    @Override
    public Configuration getConfiguration(final LoggerContext loggerContext, final String name, final URI configLocation) {
        if (!isActive()) {
            return null;
        }

        if (configLocation != null) {
            final ConfigurationSource source = ConfigurationSource.fromUri(configLocation);
            if (source != null) {
                return getConfiguration(loggerContext, source);
            }
        }

        final Configuration configuration = getConfigurationFromFile(loggerContext, name);
        return configuration != null ? configuration : getDefaultConfiguration(name);
    }

    /**
     * Find configuration files by name and extension.
     *
     * @param loggerContext The logger context
     * @param name          The name
     * @return The Configuration
     */
    private Configuration getConfigurationFromFile(final LoggerContext loggerContext, final String name) {
        for (final String type : getSupportedTypes()) {
            if (type.equals(ALL_TYPES)) {
                continue;
            }

            for (final String configName : getConfigNames(name, type)) {
                final ConfigurationSource source = ConfigurationSource.fromResource(configName, LoaderUtil.getThreadContextClassLoader());
                if (source != null) {
                    return getConfiguration(loggerContext, source);
                }
            }
        }

        return null;
    }

    /**
     * Construct a list with config file names.
     *
     * @param name The name
     * @param type The type
     * @return A list of config names
     */
    private List<String> getConfigNames(final String name, final String type) {
        final List<String> configNames = new ArrayList<>();

        if (Strings.isNotEmpty(name)) {
            configNames.add(getTestPrefix() + name + type);
            configNames.add(getDefaultPrefix() + name + type);
        }

        configNames.add(getTestPrefix() + type);
        configNames.add(getDefaultPrefix() + type);

        return List.copyOf(configNames);
    }

    /**
     * Returns the default Configuration.
     *
     * @param name The configuration name
     * @return The default Configuration
     */
    private Configuration getDefaultConfiguration(final String name) {
        final ConfigurationBuilder<BuiltConfiguration> builder = newConfigurationBuilder();
        builder.setConfigurationName(name);
        // Sets the level of the StatusLogger
        builder.setStatusLevel(Level.ERROR);

        // Return default logging configuration
        return builder.add(builder
                .newAppender("stdout", "console")
                .addAttribute("target", ConsoleAppender.Target.SYSTEM_OUT)
                .add(builder
                        .newLayout("PatternLayout")
                        .addAttribute("pattern", "%-5p|%d{yyyy-MM-dd HH:mm:ss}|%-20.20t|%-32.32c{1}|%m%n")))
                .add(builder
                        .newRootLogger(Level.INFO)
                        .add(builder.newAppenderRef("stdout")))
                .build();
    }
}
