package org.hansken.plugin.extraction.runtime.grpc.server.proxy;

import static java.lang.Math.min;
import static java.lang.Math.toIntExact;
import static java.lang.System.arraycopy;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNegative;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

/**
 * Cache implementation that makes a sequential and repeated random file read faster.
 * The cache reads blocks from gRPC and thus prevents unnecessary grpc queries.
 */
class RandomAccessDataCache {

    private final GrpcFacade _facade;
    private final long _fileSize;
    private final int _blockSize;
    private final int _cacheSize;
    private final String _traceUid;
    private final String _type;
    private final Cache<Long, byte[]> _cache;

    RandomAccessDataCache(final GrpcFacade facade, final long fileSize, final int blockSize, final int cacheSize,
                          final String traceUid, final String type) {
        _facade = argNotNull("facade", facade);
        _fileSize = argNotNegative("fileSize", fileSize);
        _blockSize = argNotNegative("blockSize", blockSize);
        _cacheSize = argNotNegative("cacheSize", cacheSize);
        _traceUid = argNotNull("_traceUid", traceUid);
        _type = argNotNull("type", type);
        _cache = Caffeine.newBuilder()
            .maximumSize(cacheSize)
            .build();
    }

    /**
     * Read bytes from cache or gRPC.
     * If the number of bytes read is greater than the cache, the bytes will be read directly from gRPC.
     * If not, a block of bytes will be fetched and cached.
     *
     * @param startPosition position to start read from
     * @param count number of bytes requested
     * @return the bytes requested
     */
    byte[] readFromCache(final long startPosition, final int count) {
        argNotNegative("startPosition", startPosition);
        argNotNegative("count", count);

        // read the data at once if the data is too large for the cache
        if (count > _cacheSize * _blockSize) {
            return _facade.readFromTraceData(startPosition, count, _traceUid, _type);
        }

        // calculate buffer size otherwise the buffer will contain empty bytes if the number of bytes remaining is less than the requested bytes
        final int bufferSize = toIntExact(min(count, _fileSize - startPosition));
        final byte[] buffer = new byte[bufferSize];
        int bytesCopied = 0;
        while (bytesCopied < count) {
            final long position = startPosition + bytesCopied;
            final int cachePosition = toIntExact(position % _blockSize); // calculate relative position in cache
            final byte[] block = getBlockFromCache(position);
            // copy the bytes from the cache block, unless the cache block contains fewer bytes than the requested number of bytes
            final int bytesToCopy = min(_blockSize - cachePosition, count - bytesCopied);

            // if bytesToCopy is greater than the fileSize - position then copy only the remaining bytes
            if (bytesToCopy + position > _fileSize) {
                arraycopy(block, cachePosition, buffer, bytesCopied, toIntExact(_fileSize - position));
                break;
            }

            arraycopy(block, cachePosition, buffer, bytesCopied, bytesToCopy);
            bytesCopied += bytesToCopy;
        }

        return buffer;
    }

    /**
     * Fill the cache with pre-existing data. The data cannot contain more bytes than the _fileSize
     *
     * @param data pre-existing data
     */
    void fillCache(final byte[] data) {
        final int size = data.length;
        if (size > _fileSize) {
            throw new IllegalArgumentException("data contains " + size + " bytes while " + _fileSize + " is allowed");
        }

        // cache only complete blocks if size < file size, otherwise there are incomplete blocks in the cache
        // which can cause an IndexOutOfBoundsException while reading
        final long bytesToRead = _fileSize == size ? size : size - (size % _blockSize);

        int position = 0;
        while (position < bytesToRead) {
            final int blockSize = min(calculateBlockSize(position), size - position); // the last block can be smaller
            final byte[] buffer = new byte[blockSize];
            arraycopy(data, position, buffer, 0, blockSize);
            _cache.put(Long.valueOf(position), buffer);
            position += blockSize;
        }
    }

    private byte[] getBlockFromCache(final long position) {
        return _cache.get(calculateCacheStartPosition(position), startPosition -> {
            final int blockSize = calculateBlockSize(startPosition);
            return _facade.readFromTraceData(startPosition, blockSize, _traceUid, _type);
        });
    }

    private int calculateBlockSize(final long position) {
        return toIntExact(min(_blockSize, _fileSize - position));
    }

    private long calculateCacheStartPosition(final long position) {
        return position - (position % _blockSize);
    }
}
