package org.hansken.plugin.extraction.runtime.grpc.server;

import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.hansken.plugin.extraction.runtime.grpc.common.VersionUtil.getApiVersion;
import static org.hansken.plugin.extraction.runtime.grpc.server.ExtractionPluginServer.DEFAULT_NUM_WORKERS;

import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Properties;
import java.util.function.Supplier;

import org.hansken.plugin.extraction.api.BaseExtractionPlugin;
import org.hansken.plugin.extraction.api.PluginInfo;

import net.sourceforge.argparse4j.ArgumentParsers;
import net.sourceforge.argparse4j.inf.ArgumentParser;
import net.sourceforge.argparse4j.inf.ArgumentParserException;
import net.sourceforge.argparse4j.inf.Namespace;

/**
 * Code to provide main() functionality to run an Extraction Plugin as standalone server.
 *
 * This helper class handles all arguments that can be set to start an Extraction Plugin as a server.
 *
 * @author Netherlands Forensic Institute
 */
public final class ExtractionPluginServerMain {

    private static final int DEFAULT_PORT_NUMBER = 8999;

    private ExtractionPluginServerMain() {
        // hidden utility class constructor
    }

    /**
     * Utility to help starting your Extraction Plugin as a server.
     * This helper class handles all arguments that can be set to start an Extraction Plugin as a server.
     * <p>
     * This means that you can create a server for an Extraction Plugin as follows:
     *
     * <code>
     * public static void main(String[] args) {
     * ExtractionPluginServerMain.runMain(YourPlugin::new, args);
     * }
     * </code>
     *
     * @param pluginSupplier Supplier that constructs a new instance of your Extraction Plugin.
     * @param args           Arguments originally passed to main()
     */
    public static void runMain(final Supplier<BaseExtractionPlugin> pluginSupplier, final String[] args) {
        try {
            run(pluginSupplier, args);
        }
        catch (final Exception e) {
            throw new IllegalStateException(e);
        }
    }

    private static void run(final Supplier<BaseExtractionPlugin> pluginSupplier, final String[] args) throws Exception {
        final Namespace arguments = parseArguments(pluginSupplier, args);
        final String outputName = arguments.getString("labels_filename");
        if (isNotBlank(outputName)) {
            writePluginInfoProperties(pluginSupplier.get().pluginInfo(), outputName);
            // DON'T call system.exit() here -- as this method is called during the plugin build in maven, system.exit() will cause the entire build to exit
            return;
        }

        try (ExtractionPluginServer plugin = runNonBlocking(pluginSupplier, arguments)) {
            try {
                plugin.blockUntilShutdown();
            }
            catch (final InterruptedException e) {
                plugin.close();
                Thread.currentThread().interrupt();
                throw e;
            }
        }
    }

    // not useful for production usage, provides a non-blocking `main` for testing purposes
    static ExtractionPluginServer runNonBlocking(final Supplier<BaseExtractionPlugin> pluginSupplier, final Namespace arguments) throws Exception {
        final Integer maximumWorkers = pluginSupplier.get().pluginInfo().resources().maximumWorkers();
        final Integer workers = maximumWorkers == null ? arguments.getInt("workers") : maximumWorkers;
        return ExtractionPluginServer.serve(arguments.getInt("port"), workers, pluginSupplier);
    }

    private static ArgumentParser createArgParser(final Supplier<BaseExtractionPlugin> pluginSupplier) {
        final PluginInfo pluginInfo = pluginSupplier.get().pluginInfo();

        final ArgumentParser parser = ArgumentParsers.newFor(pluginInfo.name()).build()
            .defaultHelp(true)
            .description(pluginInfo.description());

        parser.addArgument("-l", "--labels-filename")
            .metavar("labels-filename")
            .help("Writes plugin labels to a specified file, or to stdout if -- is passed");

        parser.addArgument("-p", "--port")
            .metavar("port")
            .type(Integer.class)
            .setDefault(DEFAULT_PORT_NUMBER)
            .help("Port where the extraction plugin will be served");

        parser.addArgument("-w", "--workers")
            .metavar("workers")
            .type(Integer.class)
            .setDefault(DEFAULT_NUM_WORKERS)
            .help("The number of concurrent workers(i.e. traces that can be processed). Default value is " + DEFAULT_NUM_WORKERS);

        return parser;
    }

    static Namespace parseArguments(final Supplier<BaseExtractionPlugin> pluginSupplier, final String[] args) {
        final ArgumentParser parser = createArgParser(pluginSupplier);
        Namespace arguments = null;
        try {
            arguments = parser.parseArgs(args);
        }
        catch (final ArgumentParserException e) {
            parser.handleError(e);
            System.exit(1);
        }
        return arguments;
    }

    private static void writePluginInfoProperties(final PluginInfo pluginInfo, final String outputName) throws IOException {
        try (OutputStream output = new FileOutputStream(outputName)) {
            writePluginInfoProperties(pluginInfo, output);
        }
    }

    static void writePluginInfoProperties(final PluginInfo pluginInfo, final OutputStream outputStream) throws IOException {
        final Properties properties = new Properties();
        properties.put("org.hansken.plugin-info.id", pluginInfo.id().toString()); //needed for docker image name
        properties.put("org.hansken.plugin-info.id-domain", pluginInfo.id().domain());
        properties.put("org.hansken.plugin-info.id-category", pluginInfo.id().category());
        properties.put("org.hansken.plugin-info.id-name", pluginInfo.id().name());
        properties.put("org.hansken.plugin-info.version", pluginInfo.pluginVersion());
        properties.put("org.hansken.plugin-info.api-version", getApiVersion());
        properties.put("org.hansken.plugin-info.description", pluginInfo.description());
        properties.put("org.hansken.plugin-info.webpage", pluginInfo.webpageUrl());
        properties.put("org.hansken.plugin-info.deferred-iterations", Integer.toString(pluginInfo.deferredIterations()));
        properties.put("org.hansken.plugin-info.matcher", pluginInfo.hqlMatcher());
        properties.put("org.hansken.plugin-info.license", pluginInfo.license());
        properties.put("org.hansken.plugin-info.maturity-level", pluginInfo.maturityLevel().name());
        properties.put("org.hansken.plugin-info.author-name", pluginInfo.author().name());
        properties.put("org.hansken.plugin-info.author-organisation", pluginInfo.author().organisation());
        properties.put("org.hansken.plugin-info.author-email", pluginInfo.author().email());

        if (pluginInfo.resources() != null) {
            final Float maxCpu = pluginInfo.resources().maximumCpu();
            properties.put("org.hansken.plugin-info.resources-max-cpu", maxCpu == null ? "" : Float.toString(maxCpu));

            final Integer maxMem = pluginInfo.resources().maximumMemory();
            properties.put("org.hansken.plugin-info.resources-max-mem", maxMem == null ? "" : Integer.toString(maxMem));
        }
        properties.store(outputStream, "Labels for extraction plugin image");
    }
}
