package org.hansken.plugin.extraction.runtime.grpc.server;

import grpc.health.v1.HealthGrpc;
import grpc.health.v1.HealthOuterClass.HealthCheckRequest;
import grpc.health.v1.HealthOuterClass.HealthCheckResponse;
import io.grpc.stub.StreamObserver;

/**
 * gRPC service that implements the gRPC Health Checking Protocol: https://github.com/grpc/grpc/blob/master/doc/health-checking.md
 * This is used to perform a readiness check for the Extraction Plugin pod in Kubernetes.
 *
 * The health service will always return SERVING, no matter the requested service. This is sufficient for letting Hansken
 * wait on Extraction Plugins that take a long time (i.e. 30 seconds) before they are ready. Note that Hansken will
 * timeout after one minute.
 *
 * @author Netherlands Forensic Institute
 */
public class HealthService extends HealthGrpc.HealthImplBase {

    /**
     * Puts the status SERVING on the response. Kubernetes will probe this service to determine the readiness of the
     * pod of the Extraction Plugin. A gRPC client (like Hansken) can wait until the pod is ready and only then continue
     * to perform other gRPC calls.
     *
     * @param request the service to be checked, currently this is unused
     * @param responseObserver a gRPC StreamObserver that handles the response
     */
    @Override
    public void check(final HealthCheckRequest request, final StreamObserver<HealthCheckResponse> responseObserver) {
        final HealthCheckResponse response = HealthCheckResponse.newBuilder()
            .setStatus(HealthCheckResponse.ServingStatus.SERVING)
            .build();
        responseObserver.onNext(response);
        responseObserver.onCompleted();
    }

    /**
     * A client can call the Watch method to perform a streaming health-check. The server will immediately send back a
     * message indicating the current serving status. It will then subsequently send a new message whenever the
     * service's serving status changes. This MVP implementation just returns SERVING and will not change status.
     * TODO: Determine health/liveness by the number of threads in use by the Extraction Plugin (HANSKEN-16909)
     *
     * @param request the service to be checked, currently this is unused
     * @param responseObserver a gRPC StreamObserver that handles the response
     */
    @Override
    public void watch(final HealthCheckRequest request, final StreamObserver<HealthCheckResponse> responseObserver) {
        check(request, responseObserver);
    }

}
