package org.hansken.plugin.extraction.runtime.grpc.server.proxy;

import static java.util.Objects.checkFromIndexSize;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotEmpty;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.io.OutputStream;

import org.hansken.plugin.extraction.api.DataWriter;
import org.hansken.plugin.extraction.api.Trace;

/**
 * Proxy implementation for the server side of the {@link OutputStream} to which raw data of a {@link Trace} is written
 * (see {@link Trace#setData(String, DataWriter)}. It delegates calls through gRPC (see {@link GrpcFacade}) to the
 * trace currently being processed on the client side.
 * <p>
 * <strong>Note:</strong> this implementation is not thread-safe.
 */
class OutputStreamProxy extends OutputStream {

    private final byte[] _buffer = new byte[1];
    private final String _id;
    private final GrpcFacade _facade;
    private final String _dataType;

    OutputStreamProxy(final String id, final String dataType, final GrpcFacade facade) {
        _id = argNotEmpty("id", id);
        _dataType = argNotEmpty("dataType", dataType);
        _facade = argNotNull("facade", facade);
        _facade.beginWritingData(_id, _dataType);
    }

    @Override
    public void write(final int b) {
        _buffer[0] = (byte) b;
        write(_buffer, 0, 1);
    }

    @Override
    public void write(final byte[] buffer, final int offset, final int length) {
        checkFromIndexSize(offset, length, buffer.length);
        _facade.writeData(_id, _dataType, buffer, offset, length);
    }

    @Override
    public void close() {
        _facade.finishWritingData(_id, _dataType);
    }
}