package org.hansken.plugin.extraction.runtime.grpc.server.proxy;

import static org.hansken.plugin.extraction.runtime.grpc.common.Checks.assertIsNotMeta;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import org.hansken.extraction.plugin.grpc.RpcDataContext;
import org.hansken.plugin.extraction.api.DataContext;
import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.RandomAccessData;

/**
 * Proxy implementation for the server side of the {@link ExtractionPlugin extraction plugin} framework. It
 * delegates calls through gRPC (see {@link GrpcFacade}) to the context on the client side.
 */
public final class ExtractionContextProxy implements DataContext, AutoCloseable {

    private final String _dataType;
    private final RandomAccessData _data;

    ExtractionContextProxy(final String dataType, final RandomAccessData data) {
        _dataType = argNotNull("dataType", dataType);
        _data = argNotNull("data", data);
    }

    /**
     * Create a new {@link ExtractionContextProxy context proxy} for the context described by the given
     * {@link RpcDataContext}, using the given {@link GrpcFacade} to delegate calls to.
     *
     * @param context the meta information of the context to mirror, such as the data type
     * @param traceId the id of the trace
     * @param facade the facade to delegate calls to
     * @return te context proxy
     */
    public static ExtractionContextProxy fromRpc(final RpcDataContext context, final String traceId, final GrpcFacade facade) {
        argNotNull("context", context);
        argNotNull("traceId", traceId);
        argNotNull("facade", facade);

        return new ExtractionContextProxy(
            context.getDataType(),
            RandomAccessDataProxy.fromRpc(context.getData(), traceId, facade)
        );
    }

    @Override
    public String dataType() {
        return _dataType;
    }

    @Override
    public RandomAccessData data() {
        assertIsNotMeta(dataType());
        return _data;
    }

    @Override
    public void close() throws Exception {
        _data.close();
    }
}
