package org.hansken.plugin.extraction.runtime.grpc.server.proxy;

import static java.lang.Math.min;
import static java.lang.Math.toIntExact;
import static java.lang.System.arraycopy;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotNegative;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import org.hansken.extraction.plugin.grpc.RpcRandomAccessDataMeta;
import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.RandomAccessData;

/**
 * Proxy implementation for the server side of the {@link ExtractionPlugin extraction plugin} framework. It
 * delegates calls through gRPC (see {@link GrpcFacade}) to the data currently being processed on the client side.
 * <p>
 * <strong>Note:</strong> this implementation is not thread-safe.
 */
public final class RandomAccessDataProxy implements RandomAccessData {
    private static final int CACHE_SIZE = 10; // Cache max 10 blocks of data
    private static final int BLOCK_SIZE = 1024 * 1024 / CACHE_SIZE; // total of 1MB is cached

    private final long _size;
    private final RandomAccessDataCache _cache;

    private long _position;
    private boolean _open;

    private RandomAccessDataProxy(final RpcRandomAccessDataMeta data, final String traceId, final GrpcFacade facade) {
        _size = argNotNegative("size", data.getSize());
        _cache = new RandomAccessDataCache(facade, _size, BLOCK_SIZE, CACHE_SIZE, traceId, data.getType());
        _open = true;
        _cache.fillCache(data.getFirstBytes().toByteArray());
    }

    /**
     * Create a new {@link RandomAccessDataProxy data proxy} for the data described by the given
     * {@link RpcRandomAccessDataMeta}, using the given {@link GrpcFacade} to delegate calls to.
     *
     * @param data the meta information about the data sequence, such as the size and position
     * @param traceId the id of the trace
     * @param facade the facade to delegate calls to
     * @return the data sequence proxy
     */
    public static RandomAccessData fromRpc(final RpcRandomAccessDataMeta data, final String traceId, final GrpcFacade facade) {
        argNotNull("data", data);
        argNotNull("traceId", traceId);
        argNotNull("facade", facade);
        return new RandomAccessDataProxy(data, traceId, facade);
    }

    @Override
    public long size() {
        return _size;
    }

    @Override
    public long position() {
        assertOpen();
        return _position;
    }

    @Override
    public void seek(final long position) {
        assertOpen();
        _position = assertValidPosition(position);
    }

    @Override
    public int read(final byte[] buffer, final int offset, final int count) {
        assertOpen();
        assertReadParameters(buffer, offset, count);

        final int shouldRead = toIntExact(min(count, size() - position()));
        final byte[] read = _cache.readFromCache(position(), shouldRead);
        final int actuallyRead = read.length;
        arraycopy(read, 0, buffer, offset, actuallyRead);
        seek(position() + actuallyRead);

        return shouldRead;
    }

    @Override
    public void close() {
        _open = false;
    }

    private void assertOpen() {
        if (!_open) {
            throw new IllegalStateException("the data has already been closed");
        }
    }

    private long assertValidPosition(final long position) {
        if (position < 0) {
            throw new IllegalArgumentException("the position must not be negative: " + position);
        }
        if (position > size()) {
            throw new IllegalArgumentException("the position must not be greater than the size of the data: " + position + " > " + size());
        }

        return position;
    }

    private void assertReadParameters(final byte[] buffer, final int offset, final int count) {
        argNotNegative("offset", offset);
        argNotNegative("count", count);

        if (offset + count > buffer.length) {
            throw new IllegalArgumentException(
                "offset + count is larger than the size of the buffer: " +
                    "offset " + offset + ", " +
                    "count " + count + ", " +
                    "buffer size " + buffer.length);
        }
    }
}
