package org.hansken.plugin.extraction.runtime.grpc.server.proxy;

import static org.hansken.plugin.extraction.runtime.grpc.common.Checks.assertIsNotMeta;
import static org.hansken.plugin.extraction.runtime.grpc.server.proxy.TraceProxy.getProperties;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.hansken.extraction.plugin.grpc.RpcRandomAccessDataMeta;
import org.hansken.extraction.plugin.grpc.RpcSearchTrace;
import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.RandomAccessData;
import org.hansken.plugin.extraction.api.SearchTrace;

/**
 * Proxy implementation for the server side of the {@link ExtractionPlugin extraction plugin} framework. It
 * delegates calls through gRPC (see {@link GrpcFacade}) to the search trace currently being processed on the client
 * side.
 * <p>
 * <strong>Note:</strong> this implementation is not thread-safe.
 */
public class SearchTraceProxy extends ImmutableTraceProxy implements SearchTrace {

    private final List<RpcRandomAccessDataMeta> _data;
    private final GrpcFacade _facade;

    SearchTraceProxy(final Set<String> types,
                     final Map<String, Object> properties,
                     final String traceId,
                     final List<RpcRandomAccessDataMeta> data,
                     final GrpcFacade facade) {
        super(types, properties, traceId);
        _data = argNotNull("data", data);
        _facade = facade;
    }

    public static SearchTraceProxy fromRpc(final RpcSearchTrace trace, final GrpcFacade facade) {
        argNotNull("trace", trace);

        final Set<String> types = new HashSet<>(trace.getTypesList());
        final Map<String, Object> properties = getProperties(trace.getPropertiesList());
        final String traceId = trace.getId();
        final List<RpcRandomAccessDataMeta> data = trace.getDataList();

        return new SearchTraceProxy(types, properties, traceId, data, facade);
    }

    @Override
    public RandomAccessData getData(final String dataType) {
        assertIsNotMeta(dataType);
        final RpcRandomAccessDataMeta rpcData = _data.stream()
            .filter(data -> data.getType().equals(dataType))
            .findFirst()
            .orElseThrow(() -> new IllegalStateException("No data of type " + dataType));

        return RandomAccessDataProxy.fromRpc(rpcData, get("uid"), _facade);
    }

    @Override
    public List<String> getDataTypes() {
        return _data.stream()
            .map(RpcRandomAccessDataMeta::getType)
            .collect(Collectors.toList());
    }
}
