package org.hansken.plugin.extraction.runtime.grpc.server.proxy;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toMap;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotEmpty;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hansken.extraction.plugin.grpc.RpcTrace;
import org.hansken.extraction.plugin.grpc.RpcTraceProperty;
import org.hansken.plugin.extraction.api.DataWriter;
import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.Trace;
import org.hansken.plugin.extraction.api.transformations.DataTransformation;
import org.hansken.plugin.extraction.runtime.grpc.common.Unpack;
import org.hansken.plugin.extraction.util.ThrowingConsumer;

/**
 * Proxy implementation for the server side of the {@link ExtractionPlugin extraction plugin} framework. It
 * delegates calls through gRPC (see {@link GrpcFacade}) to the trace currently being processed on the client side.
 * <p>
 * <strong>Note:</strong> this implementation is not thread-safe.
 */
public final class TraceProxy extends ImmutableTraceProxy implements Trace, AutoCloseable {
    private final Set<String> _types;
    private final Map<String, Object> _properties;

    private final String _id;
    private final GrpcFacade _facade;
    private final Set<String> _newTypes;
    private final Map<String, Object> _newProperties;
    private final List<Tracelet> _newTracelets;
    private final Map<String, List<DataTransformation>> _transformations;

    private boolean _open;
    private long _nextChild;

    private TraceProxy(
        final String id,
        final Set<String> types,
        final Map<String, Object> properties,
        final GrpcFacade facade
    ) {
        super(types, properties, id);
        _id = argNotNull("id", id);
        _types = new HashSet<>(argNotNull("types", types));
        _properties = new HashMap<>(argNotNull("properties", properties));
        _newTypes = new HashSet<>();
        _newProperties = new HashMap<>();
        _newTracelets = new ArrayList<>();
        _transformations = new HashMap<>();
        _facade = argNotNull("facade", facade);
        _open = true;
    }

    /**
     * Create a new {@link TraceProxy trace proxy} for the {@link Trace trace} currently being processed,
     * which is described by given {@link RpcTrace}. Calls to this proxy are delegated using the given
     * {@link GrpcFacade}.
     *
     * @param trace  metadata of the trace currently being processed, such as the types and properties
     * @param facade the facade to delegate calls to
     * @return the trace proxy
     */
    public static TraceProxy fromRpc(final RpcTrace trace, final GrpcFacade facade) {
        argNotNull("trace", trace);
        argNotNull("facade", facade);

        final Set<String> types = new HashSet<>(trace.getTypesList());
        final Map<String, Object> properties = getProperties(trace.getPropertiesList());

        return new TraceProxy(trace.getId(), types, properties, facade);
    }

    static Map<String, Object> getProperties(final List<RpcTraceProperty> properties) {
        return properties.stream()
            .collect(toMap(RpcTraceProperty::getName, property -> Unpack.primitive(property.getValue())));
    }

    @Override
    public Set<String> types() {
        assertOpen();
        return _types;
    }

    @Override
    public Trace addType(final String type) {
        assertOpen();
        _types.add(argNotEmpty("type", type));
        _newTypes.add(type);
        return this;
    }

    @Override
    public Set<String> properties() {
        assertOpen();
        return _properties.keySet();
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T> T get(final String name) {
        assertOpen();
        return (T) _properties.get(argNotEmpty("name", name));
    }

    @Override
    public Trace set(final String name, final Object value) {
        assertOpen();
        _properties.put(argNotEmpty("name", name), value);
        _newProperties.put(name, value);
        return this;
    }

    @Override
    public Trace addTracelet(final Tracelet tracelet) {
        _newTracelets.add(tracelet);
        return this;
    }

    @Override
    public Trace setData(final String dataType, final DataWriter writer) throws IOException {
        try (
            OutputStreamProxy stream = new OutputStreamProxy(_id, dataType, _facade)) {
            writer.writeTo(stream);
        }
        return this;
    }

    @Override
    public Trace setData(final String dataType, final List<DataTransformation> transformations) {
        argNotEmpty("dataType", dataType);
        argNotEmpty("transformations", transformations);
        _transformations.put(dataType, transformations);
        return this;
    }

    @Override
    public Trace newChild(final String name, final ThrowingConsumer<Trace, IOException> enrichChildCallback) throws IOException {
        assertOpen();
        final String id = incrementAndGetNextId();
        _facade.beginChild(id, name);
        try (TraceProxy child = new TraceProxy(id, emptySet(), emptyMap(), _facade)) {
            child.set("name", name);
            enrichChildCallback.accept(child);
            // close will 'store'/flush the trace
        }
        return this;
    }

    @Override
    public void close() {
        if (!_open) {
            return;
        }
        flush();
        if (!"0".equals(_id)) {
            _facade.finishChild(_id);
        }
        _open = false;
    }

    /**
     * Send all new updates on this proxy since initialization or last flush
     * to the client side.
     * <p>
     * <strong>Note:</strong> this method has default visibility because it is used for testing.
     */
    void flush() {
        if (_newTypes.isEmpty() && _newProperties.isEmpty() && _newTracelets.isEmpty() && _transformations.isEmpty()) {
            return;
        }
        // remove name property as it was already given through _facade#beginChild
        // and the properties are cleared anyway
        _newProperties.remove("name");
        _facade.enrichTrace(_id, _newTypes, _newProperties, _newTracelets, _transformations);
        _newTypes.clear();
        _newProperties.clear();
    }

    private void assertOpen() {
        if (!_open) {
            throw new IllegalStateException("the trace proxy has already been closed");
        }
    }

    private String incrementAndGetNextId() {
        return _id + "-" + _nextChild++;
    }
}