package org.hansken.plugin.extraction.test.base;

import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.util.Locale.ROOT;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

import static java.util.stream.Collectors.toUnmodifiableList;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.createDataStreamPath;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.dataStream;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.dataTypes;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.hasDataStreamOfType;
import static org.junit.jupiter.api.Assertions.assertFalse;
import static org.junit.jupiter.api.Assertions.fail;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Stream;

import org.hansken.plugin.extraction.api.DataWriter;

import nl.minvenj.nfi.flits.api.Trace;
import nl.minvenj.nfi.flits.api.result.ThrowableResult;
import nl.minvenj.nfi.flits.api.result.TraceResult;
import nl.minvenj.nfi.flits.base.DefaultResultValidator;
import nl.minvenj.nfi.flits.serialize.TraceToJson;

/**
 * Extension to the {@link DefaultResultValidator} which also validates each data stream of a
 * {@link org.hansken.plugin.extraction.api.Trace} by comparing to each expected file containing
 * a data stream of that type (see {@link DefaultPluginResultGenerator} and
 * {@link org.hansken.plugin.extraction.api.Trace#setData(String, DataWriter)}.
 */
public class DefaultPluginResultValidator extends DefaultResultValidator {

    public DefaultPluginResultValidator(final TraceToJson traceToJson) {
        super(traceToJson);
    }

    @Override
    public void validate(final ThrowableResult result, final Path inputPath) throws IOException {
        if (!PluginThrowableResult.isValidUsingCustomMatching(result, inputPath)) {
            super.validate(result, inputPath);
        }
    }

    @Override
    public void validate(final TraceResult result, final Path inputPath) throws IOException {
        super.validate(result, inputPath);

        final List<String> mismatches = findMismatches(inputPath, result.traces());
        if (!mismatches.isEmpty()) {
            fail(mismatches.stream().collect(joining(lineSeparator() + "\t", lineSeparator() + "\t", "")));
        }
    }

    private List<String> findMismatches(final Path inputPath, final List<Trace> traces) throws IOException {
        assertFalse(() -> traces.size() > 1, "Multiple traces are currently not supported");

        final List<String> mismatches = new ArrayList<>();

        for (final Trace trace : traces) {
            mismatches.addAll(findMismatches(inputPath, trace));

            for (final Trace childTrace : trace.children().collect(toUnmodifiableList())) {
                mismatches.addAll(findMismatches(inputPath, childTrace));
            }
        }

        return mismatches;
    }

    private List<String> findMismatches(final Path inputPath, final Trace trace) throws IOException {
        final String fileName = inputPath.getFileName().toString();
        // e.g. 'test.plugin.0-0.'
        final String filePrefix = fileName.substring(0, fileName.lastIndexOf('.')) + "." + trace.get("id") + ".";

        try (Stream<Path> files = Files.list(inputPath.getParent())) {
            final List<Path> expectedStreams = files
                .filter(Files::isRegularFile)
                .filter(file -> file.getFileName().toString().startsWith(filePrefix))
                .collect(toList());

            final List<String> mismatches = new ArrayList<>();
            for (final Path expected : expectedStreams) {
                final String dataType = expected.getFileName().toString().substring(filePrefix.length());
                // trace is expected to contain a data stream for this expected file
                if (!hasDataStreamOfType(trace, dataType)) {
                    mismatches.add(dataMissingMessage(trace, dataType, expected));
                }
                // the data stream must have equal contents compared to the expected file
                else if (!Arrays.equals(dataStream(trace, dataType), Files.readAllBytes(expected))) {
                    mismatches.add(dataMismatchMessage(trace, dataType, expected));
                }
            }
            for (final String dataType : dataTypes(trace)) {
                if (expectedStreams.stream().noneMatch(file -> file.toString().endsWith(dataType))) {
                    // there must not be streams on the trace for which there is no expected file
                    mismatches.add(
                        unexpectedDataMessage(trace, dataType, createDataStreamPath(inputPath, trace.get("id"), dataType))
                    );
                }
            }

            return mismatches;
        }
    }

    private String unexpectedDataMessage(final Trace trace, final String dataType, final Path dataFile) {
        return format(ROOT,
            "Trace with id '%s' contains unexpected data stream of type '%s', the expected result does not exist: %s",
            trace.get("id"),
            dataType,
            dataFile
        );
    }

    private String dataMismatchMessage(final Trace trace, final String dataType, final Path dataFile) {
        return format(ROOT,
            "Trace with id '%s' has a data stream of type '%s' which does not equal expected data as contained in: %s",
            trace.get("id"),
            dataType,
            dataFile
        );
    }

    private String dataMissingMessage(final Trace trace, final String dataType, final Path dataFile) {
        return format(ROOT,
            "Trace with id '%s' is missing expected data stream of type '%s', the expected result is located at: %s",
            trace.get("id"),
            dataType,
            dataFile
        );
    }
}
