/*
 * Copyright (c) 2021, Netherlands Forensic Institute
 * All rights reserved.
 */
package org.hansken.plugin.extraction.test.util;

import static java.util.Locale.ROOT;

import static org.hansken.plugin.extraction.main.StandaloneTestRunner.isVerboseEnabled;

import java.util.stream.Collectors;

import org.hansken.plugin.extraction.hql_lite.lang.human.matcher.HqlMatcher;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientTrace;

/**
 * This utility class contains some functions to format HQL based logging.
 */
public final class HqlLogger {
    public static final String HQL_MATCH_MESSAGE = "HQL match found for:";
    public static final String HQL_NO_MATCH_MESSAGE = "HQL did NOT match for:";
    private static final String HQL_MESSAGE = "%n%s%nWith trace:%n%s%n";

    private HqlLogger() {
    }

    /**
     * This method formats the input {@code trace} according to the {@link #HQL_MESSAGE}, and logs it to the provided logger.
     *
     * @param trace                 the input trace, same as {@link HqlMatcher#match}
     * @param dataType              data type of the stream, same as {@link HqlMatcher#match}
     * @param hqlMatcher            hql matcher query string
     * @param messagePrefix         a string that is pre-appended to the output message
     * @param verboseLoggingEnabled enable verbose logging for HQL queries
     *     <p>
     *     Example message:<br>
     *     <pre>
     *         {@code
     *             messagePrefix
     *             $dataType=raw
     *             With trace:
     *             dataType=raw
     *             types={text}
     *             properties={size=0, name=root, id=0}
     *         }
     *     </pre>
     */
    public static void logTrace(final ClientTrace trace, final String dataType,
                                final String hqlMatcher, final String messagePrefix, final boolean verboseLoggingEnabled) {
        if (verboseLoggingEnabled || isVerboseEnabled()) {
            // TODO HANSKEN-16167 Use SLF4J logger instead of System.out.println()
            System.out.println(logTrace(trace, dataType, hqlMatcher, messagePrefix));
        }
    }

    static String logTrace(final ClientTrace trace, final String dataType,
                           final String hqlMatcher, final String messagePrefix) {
        final StringBuilder builder = new StringBuilder();
        builder.append("dataType=").append(dataType).append("\n");

        final String types = String.join(", ", trace.types());
        builder.append("types={").append(types).append("}\n");

        final String properties = trace.properties().stream()
            .map(key -> key + "=" + trace.get(key).toString())
            .collect(Collectors.joining(", "));
        builder.append("properties={").append(properties).append("}\n");

        return String.format(ROOT, messagePrefix + HQL_MESSAGE, hqlMatcher, builder);
    }
}
