package org.hansken.plugin.extraction.test.serialize;

import static org.hansken.plugin.extraction.test.base.ProcessingUtil.extractFileName;
import static org.hansken.plugin.extraction.test.util.TestResourceUtil.relativePathToWorkingDir;

import static com.fasterxml.jackson.core.JsonParser.Feature.ALLOW_SINGLE_QUOTES;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.stream.Collectors;

import org.apache.logging.log4j.core.util.FileUtils;
import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.RandomAccessData;
import org.hansken.plugin.extraction.api.SearchTrace;
import org.hansken.plugin.extraction.api.Trace;
import org.hansken.plugin.extraction.api.TraceSearcher;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientDataContext;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientTrace;
import org.hansken.plugin.extraction.test.TestFrameworkException;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Utility methods for deserializing the input of an {@link ExtractionPlugin}, for use in the
 * Flits test framework.
 *
 * TODO deserialize all property types, how? (e.g. time zone, based on model?? based on string format??)
 */
public final class Deserialize {

    static final ObjectMapper OBJECT_MAPPER = new ObjectMapper().enable(ALLOW_SINGLE_QUOTES);
    private static final String ID = "id";
    private static final String NAME = "name";
    private static final String PATH = "path";
    private static final String DEFAULT_ID = "0";
    private static final String DEFAULT_NAME = "test-input-trace";
    private static final String DEFAULT_PATH = "/" + DEFAULT_NAME;
    private static final TypeReference<Map<String, Map<String, Object>>>
        TRACE_JSON_VALUE_TYPE = new TypeReference<>() {};

    private Deserialize() {
    }

    /**
     * Create a {@link Trace} described by the JSON in the file pointed to by given path.
     * <p>
     * The JSON is expected to contain a single {@code trace} field, which is an object
     * containing an object field per type. Each object field then contains the properties
     * of the trace for that type.
     *
     * @param jsonFile the file to read from
     * @return the deserialized trace
     */
    public static ClientTrace traceFromJson(final Path jsonFile) {
        final var traceProperties = readTraceFromJSON(jsonFile);
        return createTraceErrorHandler(jsonFile, () -> createTrace(traceProperties));
    }

    /**
     * Create a {@link SearchTrace} described by the JSON string.
     * <p>
     * The JSON is expected to contain a single {@code trace} field, which is an object
     * containing an object field per type. Each object field then contains the properties
     * of the trace for that type.
     *
     * @param jsonFile the file to read from
     * @return the deserialized trace
     * @throws IOException when an I/O error occurs or when given input file does not exist
     */
    public static SearchTrace searchTraceFromJson(final Path jsonFile) throws IOException {
        final var traceData = readTraceFromJSON(jsonFile);
        return createTraceErrorHandler(jsonFile, () -> createSearchTrace(traceData, jsonFile));
    }

    static Trace traceFromJson(final String json) {
        return createTraceErrorHandler(null, () -> createTrace(OBJECT_MAPPER.readValue(json, TRACE_JSON_VALUE_TYPE)));
    }

    private static Map<String, Map<String, Object>> readTraceFromJSON(final Path jsonFile) {
        testFileCanBeRead(jsonFile);
        try {
            return OBJECT_MAPPER.readValue(jsonFile.toFile(), TRACE_JSON_VALUE_TYPE);
        }
        catch (final Exception e) {
            throw new TestFrameworkException("The test framework can not read the input trace " + relativePathToWorkingDir(jsonFile) + "\n"
                + "> Most likely, the input trace file is not valid JSON.", e);
        }
    }

    /**
     * Executes the provided code, and wraps its exceptions --if thrown-- by a user-friendly message pointing clearly to the cause.
     *
     * @param traceFile Path to a file that is being deserialized. Can be null.
     * @param callable Code that is executed and of which all errors are caught and wrapped with a user friendly text.
     * @param <V> Trace type that this method returns
     * @return the result of callable
     * @throws TestFrameworkException wraps any exception thrown by callable
     */
    private static <V> V createTraceErrorHandler(final Path traceFile, final Callable<V> callable) {
        try {
            return callable.call();
        }
        catch (final Exception e) {
            final StringBuilder explanation = new StringBuilder();
            explanation.append("The test framework could not create a trace from the input trace JSON");
            if (traceFile != null) {
                explanation.append(" file ").append(relativePathToWorkingDir(traceFile));
            }
            explanation.append("\n");

            if (e instanceof TestFrameworkException) {
                // we already have a user-friendly exception message, we'll add the trace file-path to the error message
                // so the user knows which file to fix (handy for identifying the search trace that contains an error)
                explanation.append(" > ")
                    .append(e.getMessage())
                    .append("\n\n")
                    .append("Please update the trace input file and run the test again.\n")
                    .append("The extraction plugin SDK documentation describes in more detail how test traces and data can be defined.");
                throw new TestFrameworkException(explanation.toString(), e.getCause());
            }
            else {
                explanation.append("Most likely, the trace JSON contains incorrect invalid values.\n")
                    .append("The extraction plugin SDK documentation describes in more detail how trace files can be defined.\n")
                    .append("Please update the trace input file and run the test again.");
                throw new TestFrameworkException(explanation.toString(), e);
            }
        }
    }

    private static ClientTrace createTrace(final Map<String, Map<String, Object>> deserialized) {
        final Set<String> types = new HashSet<>();
        final Map<String, Object> properties = new HashMap<>();
        final Map<String, Object> inputTrace = parseInputTrace(deserialized, types, properties);

        // Use the types and properties parsed in parseInputTrace to create the test trace
        final String inputTraceId = readStringIntrinsicProperty(ID, DEFAULT_ID, inputTrace);
        final String inputTraceName = readStringIntrinsicProperty(NAME, DEFAULT_NAME, inputTrace);
        final String inputTracePath = readStringIntrinsicProperty(PATH, DEFAULT_PATH, inputTrace);
        return new TestTrace(inputTraceId, inputTraceName, inputTracePath, types, properties);
    }

    private static SearchTrace createSearchTrace(final Map<String, Map<String, Object>> deserialized, final Path tracePath) {
        final Set<String> types = new HashSet<>();
        final Map<String, Object> properties = new HashMap<>();
        final Map<String, Object> inputTrace = parseInputTrace(deserialized, types, properties);
        final Map<String, RandomAccessData> data = getAllData(getDataPaths(tracePath));
        // Use the types and properties parsed in parseInputTrace to create the test trace

        return new TestSearchTrace(
            (String) inputTrace.getOrDefault(ID, pathToTraceId(tracePath)),
            types,
            properties,
            data
        );
    }

    private static String pathToTraceId(final Path path) {
        return String.valueOf(
            path.toAbsolutePath().normalize().hashCode()
        );
    }

    private static String readStringIntrinsicProperty(final String property, final String defaultValue, final Map<String, Object> inputTrace) {
        // read property from inputTrace, if the value is not a String, convert the exception to something better understandable by plugin developers
        try {
            return (String) inputTrace.getOrDefault(property, defaultValue);
        }
        catch (final ClassCastException e) {
            final String actualClass = inputTrace.get(property).getClass().getSimpleName();
            throw new TestFrameworkException("The trace property '" + property + "' must be of type String, but is of type " + actualClass + ".");
        }
    }

    private static void addProperties(final Map<String, Object> properties, final String type, final Map<String, Object> value) {
        value.forEach((propertyName, propertyValue) -> {
            final String fullPropertyName = type + "." + propertyName;
            if (propertyValue instanceof Map) {
                // flatten nested map properties TODO fix/check in HANSKEN-14483
                addProperties(properties, fullPropertyName, (Map<String, Object>) propertyValue);
                return;
            }
            properties.put(fullPropertyName, propertyValue);
        });
    }

    /**
     * Create a context from a {@link RandomAccessFileData}.
     *
     * @param dataFile path to the data
     * @return the created data context
     * @throws IOException when the path is incorrect
     */
    public static ClientDataContext contextFromRandomAccessData(final Path dataFile) throws IOException {
        testFileCanBeRead(dataFile);
        try {
            return new TestContext(dataFile);
        }
        catch (final TestFrameworkException e) {
            // somewhere else we have created a more clear exception
            throw e;
        }
        catch (final Exception e) {
            throw new TestFrameworkException("The test framework is unable to create test context for data file " + relativePathToWorkingDir(dataFile), e);
        }
    }

    /**
     * Create a trace searcher with files residing in the directory pointed to by the given path.
     *
     * @param tracePath the directory whose children will be used
     * @return the created trace searcher
     */
    public static TraceSearcher traceSearcherFromData(final Path tracePath) {
        return new TestTraceSearcher(tracePath);
    }

    public static String extractDataType(final Path dataFilePath) {
        final String fileName = dataFilePath.toString();
        return fileName.substring(fileName.lastIndexOf('.') + 1);
    }

    private static Map<String, Object> parseInputTrace(final Map<String, Map<String, Object>> deserialized,
                                                       final Set<String> types, final Map<String, Object> properties) {
        if (!deserialized.containsKey("trace")) {
            throw new TestFrameworkException("The trace JSON does not contain a trace object.");
        }

        final Map<String, Object> inputTrace = deserialized.get("trace");
        // Fills types and properties from the parsed contents of the input trace
        inputTrace.forEach((type, value) -> {
            if (value instanceof Map) {
                // In case it is not a meta property, it is a complex object which contains multiple properties
                types.add(type);
                addProperties(properties, type, (Map<String, Object>) value);
            }
            else {
                // if it is a meta-property, simply add it to the list of properties
                properties.put(type, value);
            }
        });

        return inputTrace;
    }

    private static Map<String, RandomAccessData> getAllData(final List<Path> dataPaths) {
        return dataPaths.stream().collect(Collectors.toMap(Deserialize::extractDataType, Deserialize::getData));
    }

    private static List<Path> getDataPaths(final Path tracePath) {
        final String traceFileName = extractFileName(tracePath);
        return Arrays.stream(tracePath.getParent().toFile().listFiles()) // Get all files in directory
            .filter(file -> !FileUtils.getFileExtension(file).equals("trace")) // Filter all non-trace files
            .map(File::toPath) // Convert to Path
            .filter(path -> extractFileName(path).equals(traceFileName)) // Filter all same name files
            .collect(Collectors.toList());
    }

    private static RandomAccessData getData(final Path path) {
        testFileCanBeRead(path);
        try {
            return new RandomAccessFileData(path.toFile());
        }
        catch (final IOException e) {
            throw new TestFrameworkException("The test framework is unable to read the trace data file " + relativePathToWorkingDir(path), e);
        }
    }

    private static void testFileCanBeRead(final Path file) {
        if (!file.toFile().exists()) {
            throw new TestFrameworkException("The test framework is unable to open the test file " + relativePathToWorkingDir(file) + "\n"
                + "> It seems the file does not exist.");
        }
        if (!file.toFile().isFile()) {
            throw new TestFrameworkException("The test framework is unable to open the test file " + relativePathToWorkingDir(file) + "\n"
                + "> It seems the file is not an actual file.");
        }
        if (!file.toFile().canRead()) {
            throw new TestFrameworkException("The test framework is unable to open the test file " + relativePathToWorkingDir(file) + "\n"
                + "> Most likely, the input trace file has incorrect file permissions.");
        }
    }
}
