package org.hansken.plugin.extraction.test.base;

import static java.util.stream.Collectors.toList;

import static org.hansken.plugin.extraction.test.base.ProcessingUtil.createDataStreamPath;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.dataStream;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.dataTypes;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import org.hansken.plugin.extraction.api.DataWriter;

import nl.minvenj.nfi.flits.api.Trace;
import nl.minvenj.nfi.flits.api.result.TraceResult;
import nl.minvenj.nfi.flits.base.DefaultResultGenerator;
import nl.minvenj.nfi.flits.serialize.TraceToJson;

/**
 * Extension to the {@link DefaultPluginResultGenerator} which also writes each data stream of a
 * {@link org.hansken.plugin.extraction.api.Trace} to a separate file (see
 * {@link org.hansken.plugin.extraction.api.Trace#setData(String, DataWriter)}).
 */
public class DefaultPluginResultGenerator extends DefaultResultGenerator {

    public DefaultPluginResultGenerator(final TraceToJson traceToJson) {
        super(traceToJson);
    }

    @Override
    public void generate(final TraceResult result, final Path outputPath) throws IOException {
        super.generate(result, outputPath);
        writeDataStreams(outputPath, result.traces());
    }

    private void writeDataStreams(final Path outputPath, final List<Trace> traces) throws IOException {
        for (final Trace trace : traces) {
            for (final String dataType : dataTypes(trace)) {
                final Path dataFile = createDataStreamPath(outputPath, trace.get("id"), dataType);
                if (!Files.exists(dataFile)) {
                    // accept the possible race condition
                    Files.createDirectories(dataFile.getParent());
                }
                Files.write(dataFile, dataStream(trace, dataType));
            }

            writeDataStreams(outputPath, trace.children().collect(toList()));
        }
    }
}
