package org.hansken.plugin.extraction.test.serialize;

import static org.hansken.plugin.extraction.test.serialize.Deserialize.OBJECT_MAPPER;
import static org.hansken.plugin.extraction.test.util.Constants.DATA_STREAM_NAME_PREFIX;
import static org.hansken.plugin.extraction.test.util.Constants.TRANSFORMATIONS_PREFIX;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;

import org.hansken.plugin.extraction.api.DataWriter;
import org.hansken.plugin.extraction.api.Trace;
import org.hansken.plugin.extraction.api.transformations.DataTransformation;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientTrace;
import org.hansken.plugin.extraction.util.ThrowingConsumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonProcessingException;

/**
 * Simple {@link Trace} and {@link ClientTrace} implementation used as input for
 * a Flits framework test run. It does no validation whatsoever.
 */
public final class TestTrace implements ClientTrace {

    private static final Logger LOG = LoggerFactory.getLogger(TestTrace.class);

    private final AtomicInteger _nextChildId;
    private final AtomicInteger _nextTraceletId;
    private final Set<String> _types;
    private final Map<String, Object> _properties;

    public TestTrace(final String name) {
        this("0", name, "/" + name);
    }

    public TestTrace(final String id, final String name, final String path) {
        _nextChildId = new AtomicInteger(0);
        _nextTraceletId = new AtomicInteger(0);
        _types = new HashSet<>();
        _properties = new HashMap<>();
        set("id", id).set("name", name).set("path", path);
    }

    public TestTrace(final String id, final String name, final String path, final Set<String> types, final Map<String, Object> properties) {
        _nextChildId = new AtomicInteger(0);
        _nextTraceletId = new AtomicInteger(0);
        _types = types;
        _properties = properties;
        _properties.put("id", id);
        _properties.put("name", name);
        _properties.put("path", path);
    }

    @Override
    public String traceId() {
        return "0";
    }

    @Override
    public Set<String> types() {
        return _types;
    }

    @Override
    public Trace addType(final String type) {
        _types.add(type);
        return this;
    }

    @Override
    public Set<String> properties() {
        return _properties.keySet();
    }

    @Override
    public <T> T get(final String name) {
        return (T) _properties.get(name);
    }

    @Override
    public Trace set(final String name, final Object value) {
        if (name.startsWith(DATA_STREAM_NAME_PREFIX)) {
            throw new IllegalArgumentException("cannot set a property name prefixed with " + DATA_STREAM_NAME_PREFIX);
        }
        else if (name.startsWith(TRANSFORMATIONS_PREFIX)) {
            throw new IllegalArgumentException("cannot set a property name prefixed with " + TRANSFORMATIONS_PREFIX);
        }
        changeProperty(name, value);
        return this;
    }

    @Override
    public Trace addTracelet(final Tracelet tracelet) {
        final int traceletId = _nextTraceletId.incrementAndGet();
        final String propertyName = tracelet.getName();
        addType(propertyName);
        tracelet.getValue().forEach(traceletProperty -> {
            changeProperty(propertyName + "." + traceletId + "." + traceletProperty.getName(),
                traceletProperty.getValue());
        });
        return this;
    }

    @Override
    public Trace setData(final String dataType, final DataWriter writer) throws IOException {
        final ByteArrayOutputStream output = new ByteArrayOutputStream();
        writer.writeTo(output);
        changeProperty(DATA_STREAM_NAME_PREFIX + dataType, output.toByteArray());
        return this;
    }

    @Override
    public Trace setData(final String dataType, final List<DataTransformation> transformations) {
        try {
            final String serializedTransformations = OBJECT_MAPPER.writeValueAsString(transformations);
            changeProperty(TRANSFORMATIONS_PREFIX + dataType + ".descriptor", serializedTransformations);
        }
        catch (final JsonProcessingException e) {
            LOG.error("Could not serialize transformation.", e);
        }
        return this;
    }

    @Override
    public Trace newChild(final String name, final ThrowingConsumer<Trace, IOException> enrichChildCallback) throws IOException {
        enrichChildCallback.accept(new TestTrace(get("id") + "-" + _nextChildId.getAndIncrement(), name, get("path") + "/" + name));
        return this;
    }

    @Override
    public String name() {
        return get("name");
    }

    @Override
    public TestTrace newChild(final String name) {
        return new TestTrace(get("id") + "-" + _nextChildId.getAndIncrement(), name, get("path") + "/" + name);
    }

    @Override
    public void save() throws Exception {
        // NO-OP, no save() action required for this test-runner
    }

    /**
     * Changes a property if this is allowed according to the Hansken rules.
     * <p></p>
     * <b>NOTE: audit validations are not performed.</b>
     *
     * @param propertyName  the name of the property key-value pair
     * @param propertyValue the value of the property key-value pair
     * @throws IllegalStateException if this propertyName is already set
     * @see <a href="https://www.atlassian-ext.holmes.nl/bitbucket/pr ojects/HANSKEN/repos/hansken/browse/projects/services/trace-service/api/src/main/java/nl/minvenj/nfi/hansken/trace/validate/ValidatingTraceBuilder.java#250">Hansken trace validation</a>
     */
    private void changeProperty(final String propertyName, final Object propertyValue) {
        final Object valueOld = _properties.get(propertyName);
        if (valueOld != null && !(valueOld instanceof Collection)) {
            throw new IllegalStateException("property " + propertyName + " already set");
        }
        _properties.put(propertyName, propertyValue);
    }
}
