// TODO HANSKEN-14108: when FLITS support is integrated in Hansken, this should be removed
// This is a copy of the TraceToJson from FLITS, which is patched here waiting for the resolution
// mentioned in the issue above.
package nl.minvenj.nfi.flits.serialize;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Comparator.comparingInt;
import static java.util.Map.entry;
import static java.util.Map.ofEntries;
import static java.util.Spliterator.ORDERED;
import static java.util.Spliterators.spliteratorUnknownSize;
import static java.util.function.Function.identity;
import static java.util.function.Predicate.not;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toSet;
import static java.util.stream.StreamSupport.stream;

import static org.apache.commons.lang3.StringUtils.substringAfter;
import static org.hansken.plugin.extraction.test.util.Constants.DATA_STREAM_NAME_PREFIX;
import static org.hansken.plugin.extraction.test.util.Constants.TRANSFORMATIONS_PREFIX;

import static nl.minvenj.nfi.common.argchecks.ArgChecks.argNotEmpty;
import static nl.minvenj.nfi.common.argchecks.ArgChecks.argNotNull;
import static nl.minvenj.nfi.flits.serialize.MapUtil.getMapNamePostfix;
import static nl.minvenj.nfi.flits.serialize.MapUtil.mapMap;
import static nl.minvenj.nfi.flits.serialize.PatternMatch.when;
import static nl.minvenj.nfi.flits.util.ExceptionUtil.uncheckedIO;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.hansken.plugin.extraction.api.Vector;

import com.fasterxml.jackson.core.JsonEncoding;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.util.DefaultIndenter;
import com.fasterxml.jackson.core.util.DefaultPrettyPrinter;

import nl.minvenj.nfi.flits.api.Trace;

/**
 * Utility class for creating a JSON representation of {@link Trace traces}. See
 * {@link #toJsonString(List)} for more information.
 */
public final class TraceToJson {

    private static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ISO_OFFSET_DATE_TIME.withZone(ZoneOffset.UTC);
    private static final DefaultIndenter ARRAY_INDENTER = new DefaultIndenter("  ", "\n");
    private static final CustomOrder DEFAULT_ORDER = CustomOrder.create(any -> true, String::compareTo);
    private static final Map<String, Integer> TIMESTAMP_FIELD_ORDER = ofEntries(
        entry("format", 0),
        entry("valid", 1),
        entry("timeZone", 2),
        entry("resolution", 3),
        entry("year", 4),
        entry("month", 5),
        entry("day", 6),
        entry("hour", 7),
        entry("minute", 8),
        entry("second", 9),
        entry("millisecond", 10),
        entry("microsecond", 11),
        entry("nanosecond", 12),
        entry("invalidUnits", 13)
    );
    private static final CustomOrder DEFAULT_TIMESTAMP_ORDER = CustomOrder.create(
        trace -> trace.types().contains("timestamp"),
        comparingInt(property -> TIMESTAMP_FIELD_ORDER.getOrDefault(timestampProperty(property), Integer.MAX_VALUE))
    );

    private final List<BiPredicate<String, Object>> _exclusionFilters;
    private final List<CustomConversion> _customConversions;
    private final List<CustomOrder> _customOrders;

    private TraceToJson(final List<BiPredicate<String, Object>> exclusionFilters,
                        final List<CustomConversion> customConversions,
                        final List<CustomOrder> customOrders) {
        _exclusionFilters = toImmutableList(argNotNull("exclusionFilters", exclusionFilters));
        _customConversions = toImmutableList(argNotNull("customConversions", customConversions));
        _customOrders = toImmutableList(argNotNull("customOrders", customOrders));
    }

    /**
     * Create a new instance. By default, it has a single {@link TraceToJson#addCustomOrder(Predicate, Comparator)
     * custom order} defined, which triggers on a {@link Trace trace} with type {@code timestamp}. It orders
     * the properties in a 'logical/natural' order.
     * <p>
     * The default fallback for all other traces is sorting them by property name.
     *
     * @return a default trace serializer instance
     */
    public static TraceToJson create() {
        return new TraceToJson(
            singletonList((property, value) -> false),
            emptyList(),
            singletonList(DEFAULT_TIMESTAMP_ORDER)
        );
    }

    /**
     * Add a filter for excluding properties in the serialized {@link Trace trace} output,
     * based on the name or the value of the property.
     * <p>
     * For example, if you want to ignore all properties ending on {@code ".data"},
     * or all properties of type {@code byte[]}:
     * <pre>
     * {@code
     *   final TraceToJson traceToJson = TraceToJson.create()
     *       .exclude((name, value) -> name.endsWith(".data") || value instanceof byte[])
     * }
     * </pre>
     *
     * @param exclusionFilter the filter to add
     * @return a new TraceToJson based on {@code this}, extended with the filter
     */
    public TraceToJson exclude(final BiPredicate<String, Object> exclusionFilter) {
        argNotNull("exclusionFilter", exclusionFilter);
        final List<BiPredicate<String, Object>> exclusionFilters = new ArrayList<>(_exclusionFilters);
        exclusionFilters.add(exclusionFilter);
        return new TraceToJson(exclusionFilters, _customConversions, _customOrders);
    }

    /**
     * Add a custom conversion for elements of a given type. A conversion can convert an element
     * of the given type to a {@link String}. Multiple conversions can be added by calling this method
     * repeatedly.
     * <p>
     * If an element is matched, the name of the property corresponding to the element will be added
     * as the field name, the conversion of the element will be added as the field value.
     * <p>
     * <strong>Note:</strong> these conversions take precedence before the default implementations, so
     * for example adding a conversion for a {@link Long} type will override the existing behaviour.
     *
     * @param type the type class to match with
     * @param conversion the conversion to apply
     * @param <T> the required type
     * @return a new TraceToJson based on {@code this}, extended with the custom conversion
     */
    public <T> TraceToJson addCustomConversion(final Class<T> type, final Function<? super T, String> conversion) {
        argNotNull("type", type);
        argNotNull("conversion", conversion);
        final List<CustomConversion> customConversions = new ArrayList<>(_customConversions);
        customConversions.add(CustomConversion.create(type, conversion));
        return new TraceToJson(_exclusionFilters, customConversions, _customOrders);
    }

    /**
     * Add a custom order for the properties of a {@link Trace trace}. If a trace is matched, the properties
     * will be sorted using the supplied comparator.
     * <p>
     * <strong>Note:</strong>the custom orders will take precedence before the already supplied orders, and
     * before the by the default ordering rules (see {@link TraceToJson#create()}).
     *
     * @param predicate the predicate to check if the order applies to a certain trace
     * @param comparator the comparator to use
     * @return a new TraceToJson based on {@code this}, extended with the custom order
     */
    public TraceToJson addCustomOrder(final Predicate<? super Trace> predicate, final Comparator<? super String> comparator) {
        argNotNull("predicate", predicate);
        argNotNull("comparator", comparator);
        final List<CustomOrder> customOrders = new LinkedList<>(_customOrders);
        customOrders.add(0, CustomOrder.create(predicate, comparator));
        return new TraceToJson(_exclusionFilters, _customConversions, customOrders);
    }

    /**
     * Create a JSON string from a list of {@link Trace trace}. The traces tree will be traversed
     * and each property name and corresponding value will be added as a field to the
     * JSON object. The properties will be sorted lexicographically by name. The children
     * which will always come after all other properties. They will however also be sorted
     * in the same manner, by type.
     * <p>
     * Each type of property has a certain conversion defined by this class. As an example,
     * a {@link Map} will be converted to a JSON object, with keys and values as the fields.
     * <p>
     * See the implementation for the conversion of other types.
     *
     * @param traces the traces to create a JSON representation from
     * @return the traces as a JSON string
     * @throws UncheckedIOException when an I/O error occurs, e.g., during reading of a data stream property
     *                              or writing to the JSON stream
     */
    public String toJsonString(final List<Trace> traces) {
        argNotNull("traces", traces);
        final ByteArrayOutputStream output = new ByteArrayOutputStream();
        writeToStream(traces, output);
        return output.toString(UTF_8);
    }

    /**
     * Write a JSON representation of a list of {@link Trace} to a {@link File}, see
     * {@link #toJsonString(List)} for more information about the conversion to a JSON string.
     * <p>
     * If the file does not exists, it will be created first.
     *
     * @param traces the traces to create a JSON representation from
     * @param file  the file to write to
     * @throws UncheckedIOException when an I/O error occurs, e.g., during reading of data stream property
     */
    public void writeToFile(final List<Trace> traces, final File file) {
        argNotNull("traces", traces);
        argNotNull("file", file);

        uncheckedIO(() -> {
            file.createNewFile();
            writeToStream(traces, new FileOutputStream(file));
        });
    }

    /**
     * Write a JSON representation of a list of {@link Trace} to a {@link Path}, see
     * {@link #toJsonString(List)} for more information about the conversion to a JSON string.
     * <p>
     * If the path does not exists, it will be created first.
     *
     * @param traces the traces to create a JSON representation from
     * @param path  the path to write to
     * @throws UncheckedIOException when an I/O error occurs, e.g., during reading of data stream property
     */
    public void writeToPath(final List<Trace> traces, final Path path) {
        argNotNull("traces", traces);
        argNotNull("path", path);
        uncheckedIO(() -> {
            final Path parent = path.getParent();
            if (!Files.exists(parent)) {
                Files.createDirectories(parent);
            }
            writeToFile(traces, path.toFile());
        });
    }

    /**
     * Write a JSON representation of a list of {@link Trace trace} to an {@link OutputStream}, see
     * {@link #toJsonString(List)} for more information about the conversion to a JSON string.
     *
     * @param traces the traces to create a JSON representation from
     * @param outputStream the output stream to write to
     * @throws UncheckedIOException when an I/O error occurs, e.g., during reading of a data stream property
     *                              or writing to the output stream
     */
    public void writeToStream(final List<Trace> traces, final OutputStream outputStream) {
        argNotNull("traces", traces);
        argNotNull("outputStream", outputStream);

        final DefaultPrettyPrinter prettyPrinter = new PlatformIndependentPrettyPrinter();
        prettyPrinter.indentArraysWith(ARRAY_INDENTER);

        final JsonFactory factory = new JsonFactory();
        final JsonGenerator generator = uncheckedIO(() -> factory.createGenerator(outputStream, JsonEncoding.UTF8));
        generator.setPrettyPrinter(prettyPrinter);

        writeToGenerator(traces, generator);
    }

    /**
     * Write a JSON representation of a list of {@link Trace trace} to a {@link JsonGenerator}, see
     * {@link #toJsonString(List)} for more information about the conversion to a JSON string. If the list is of size 1 {@link #writeToGenerator(Trace, JsonGenerator)} will be called.
     *
     * @param traces    the traces to create a JSON representation from
     * @param generator the JSON generator to write to
     * @throws IllegalArgumentException when multiple traces are given. Multiple traces are currently not supported.
     */
    public void writeToGenerator(final List<Trace> traces, final JsonGenerator generator) {
        argNotNull("traces", traces);
        argNotEmpty("traces", traces);
        argNotNull("generator", generator);

        if (traces.size() == 1) {
            writeToGenerator(traces.get(0), generator);
        }
        else {
            throw new IllegalArgumentException("Multiple traces are currently not supported");
        }
    }

    private void writeToGenerator(final Trace trace, final JsonGenerator generator) {
        argNotNull("trace", trace);
        argNotNull("generator", generator);
        uncheckedIO(() -> {
            generator.writeStartObject();

            writeTypes(trace, generator);
            writeChildren(trace, generator);
            writeData(trace, generator);

            generator.writeEndObject();
            generator.flush();
        });
    }

    public void writeTypes(final Trace trace, final JsonGenerator generator) {
        uncheckedIO(() -> {
            generator.writeObjectFieldStart("trace");
            generator.writeStringField("id", trace.get("id"));
            generator.writeStringField("name", trace.get("name"));
            generator.writeStringField("path", trace.get("path"));

            final Map<String, Set<Property>> propertiesByType = trace.properties().stream()
                .filter(not("id"::equals))
                .filter(not("name"::equals))
                .filter(not("path"::equals))
                .filter(not("image"::equals))
                .filter(not(property -> property.startsWith(DATA_STREAM_NAME_PREFIX)))
                .map(Property::parse)
                .collect(groupingBy(Property::type, TreeMap::new, mapping(identity(), toSet())));

            trace.types().stream().sorted().forEachOrdered(type -> uncheckedIO(() -> {
                generator.writeObjectFieldStart(type);
                writeProperties(trace, propertiesByType.getOrDefault(type, emptySet()), generator);
                generator.writeEndObject();
            }));

            writePreviews(trace, generator);

            generator.writeEndObject();
        });
    }

    public void writeProperties(final Trace trace, final Set<Property> properties, final JsonGenerator generator) {
        final Map<Property, Object> mapProperties = properties.stream()
            .filter(property -> property.name().contains("."))
            .collect(Collectors.toMap(
                property -> Property.parse(getMapNamePostfix(property)), // new property without type prefix
                property -> trace.get(property.fullName()))
            );
        final Map<String, Object> mapPropertyValues = mapMap(mapProperties, 0);
        final Set<Property> newMapProperties = mapPropertyValues.keySet().stream()
            .map(Property::parse)
            .collect(Collectors.toSet());

        final Set<Property> nonMapProperties = properties.stream()
            .filter(property -> !property.name().contains("."))
            .collect(Collectors.toUnmodifiableSet());
        final Map<String, Object> nonMapPropertyValues = nonMapProperties.stream()
            .collect(Collectors.toMap(Property::fullName, property -> trace.get(property.fullName())));

        final Map<String, Object> allPropertyValues = new HashMap<>();
        allPropertyValues.putAll(mapPropertyValues);
        allPropertyValues.putAll(nonMapPropertyValues);

        final Set<Property> allProperties = new HashSet<>();
        allProperties.addAll(newMapProperties);
        allProperties.addAll(nonMapProperties);

        _customOrders.stream()
            .filter(customOrder -> customOrder.triggersOn(trace))
            .findFirst()
            .orElse(DEFAULT_ORDER)
            .order(allProperties)
            .forEachOrdered(property -> write(allPropertyValues, property, generator));
    }

    private void writeChildren(final Trace trace, final JsonGenerator generator) throws IOException {
        final Iterator<Trace> iterator = trace.children().iterator();
        // do not write empty children array  if there are none
        if (iterator.hasNext()) {
            generator.writeArrayFieldStart("children");

            stream(spliteratorUnknownSize(iterator, ORDERED), false)
                .sorted(comparingInt((Trace child) -> child.types().size()).thenComparingInt(child -> child.types().hashCode()))
                .forEachOrdered(child -> writeToGenerator(child, generator));

            generator.writeEndArray();
        }
    }

    private void writeData(final Trace trace, final JsonGenerator generator) {
        final Map<String, Set<Property>> transformationPropertiesByType = trace.properties().stream()
            .filter(property -> property.startsWith(TRANSFORMATIONS_PREFIX))
            .map(Property::parse)
            .collect(groupingBy(Property::type, TreeMap::new, mapping(identity(), toSet())));

        // do not write empty data field if there are no transformations
        if (!transformationPropertiesByType.isEmpty()) {
            uncheckedIO(() -> {
                generator.writeObjectFieldStart("data");

                transformationPropertiesByType.keySet().stream()
                    .sorted()
                    .map(type -> type.substring(1))
                    .forEachOrdered(type -> uncheckedIO(() -> {
                        generator.writeObjectFieldStart(type);
                        writeProperties(trace, transformationPropertiesByType.getOrDefault(TRANSFORMATIONS_PREFIX + type, emptySet()), generator);
                        generator.writeEndObject();
                    }));

                generator.writeEndObject();
            });
        }
    }

    public void writePreviews(final Trace trace, final JsonGenerator generator) {
        uncheckedIO(() -> {
            final List<String> previewProperties = trace.properties().stream()
                .filter(property -> property.startsWith("preview."))
                .sorted().collect(Collectors.toList());
            // create map if multiple previews are present
            if (previewProperties.size() > 1) {
                generator.writeObjectFieldStart("preview");
            }
            for (final String previewProperty : previewProperties) {
                write(getPreviewPropertyName(previewProperties.size(), previewProperty), trace.get(previewProperty), generator);
            }
            if (previewProperties.size() > 1) {
                generator.writeEndObject();
            }
        });
    }

    private static String getPreviewPropertyName(final int numberOfPreviews, final String previewProperty) {
        return numberOfPreviews > 1 ? substringAfter(previewProperty, "preview.") : previewProperty;
    }

    private void write(final Map<String, Object> tracePropertyValues, final Property property, final JsonGenerator generator) {
        final Object value = tracePropertyValues.get(property.fullName());
        write(property.name(), value, generator);
    }

    private void write(final String name, final Object value, final JsonGenerator generator) {
        for (final BiPredicate<String, Object> filter : _exclusionFilters) {
            if (filter.test(name, value)) {
                return;
            }
        }
        when(value)
            .matchesAny(_customConversions, string -> writeString(name, string, generator))
            .matches(Boolean.class, boolVal -> writeBoolean(name, boolVal, generator))
            .matches(Date.class, date -> writeDate(name, date, generator))
            .matches(Double.class, doubleVal -> writeDouble(name, doubleVal, generator))
            .matches(Integer.class, intVal -> writeInteger(name, intVal, generator))
            .matches(Long.class, longVal -> writeLong(name, longVal, generator))
            .matches(Map.class, map -> writeMap(name, map, generator))
            .matches(Stream.class, stream -> writeStream(name, stream, generator))
            .matches(String.class, string -> writeString(name, string, generator))
            .matches(Supplier.class, supplier -> writeSupplier(name, supplier, generator))
            .matches(Trace.class, trace -> writeTrace(name, trace, generator))
            .matches(Vector.class, vector -> writeVector(name, vector, generator))
            .matches(Collection.class, collection -> writeStream(name, collection.stream(), generator))
            .elseThrow(unknownProperty -> new IllegalArgumentException("unhandled type of property [" + name + "]: " + unknownProperty.getClass()));
    }

    private static void writeBoolean(final String name, final Boolean value, final JsonGenerator generator) {
        uncheckedIO(() -> generator.writeBooleanField(name, value));
    }

    private static void writeDate(final String name, final Date value, final JsonGenerator generator) {
        final Instant instant = value.toInstant();
        final String dateTimeString = DATE_TIME_FORMATTER.format(instant);
        uncheckedIO(() -> generator.writeStringField(name, dateTimeString));
    }

    private static void writeDouble(final String name, final Double value, final JsonGenerator generator) {
        uncheckedIO(() -> generator.writeNumberField(name, value));
    }

    private static void writeInteger(final String name, final Integer value, final JsonGenerator generator) {
        uncheckedIO(() -> generator.writeNumberField(name, value));
    }

    private static void writeLong(final String name, final Long value, final JsonGenerator generator) {
        uncheckedIO(() -> generator.writeNumberField(name, value));
    }

    private void writeMap(final String name, final Map<?, ?> value, final JsonGenerator generator) {
        uncheckedIO(() -> {
            generator.writeObjectFieldStart(name);
            new TreeMap<>(value).forEach((k, v) -> {
                // if (!(k instanceof String && v instanceof String)) {
                //     throw new IllegalArgumentException("unhandled type of key or value: " + k.getClass() + ", " + v.getClass());
                // }
                write((String) k, v, generator);
            });
            generator.writeEndObject();
        });
    }

    private void writeStream(final String name, final Stream<?> value, final JsonGenerator generator) {
        uncheckedIO(() -> {
            generator.writeArrayFieldStart(name);
            value.forEachOrdered(element -> writeArrayElement(element, generator));
            generator.writeEndArray();
        });
    }

    private static void writeString(final String name, final String value, final JsonGenerator generator) {
        uncheckedIO(() -> generator.writeStringField(name, value));
    }

    private static void writeVector(final String name, final Vector value, final JsonGenerator generator) {
        uncheckedIO(() -> generator.writeStringField(name, value.toBase64()));
    }

    private void writeSupplier(final String name, final Supplier<?> supplier, final JsonGenerator generator) {
        write(name, supplier.get(), generator);
    }

    private void writeArrayElement(final Object element, final JsonGenerator generator) {
        when(element)
            .matchesAny(_customConversions, string -> writeStringEntry(string, generator))
            .matches(String.class, string -> writeStringEntry(string, generator))
            .matches(Long.class, longValue -> writeLongEntry(longValue, generator))
            .elseThrow(unknownProperty -> new IllegalArgumentException("unhandled type of stream element: " + element.getClass()));
    }

    private void writeTrace(final String name, final Trace trace, final JsonGenerator generator) {
        uncheckedIO(() -> {
            generator.writeFieldName(name);
            writeToGenerator(trace, generator);
        });
    }

    private static void writeStringEntry(final String value, final JsonGenerator generator) {
        uncheckedIO(() -> generator.writeString(value));
    }

    private static void writeLongEntry(final long value, final JsonGenerator generator) {
        uncheckedIO(() -> generator.writeNumber(value));
    }

    private static <T> List<T> toImmutableList(final List<T> list) {
        return List.copyOf(list);
    }

    private static String timestampProperty(final String property) {
        return property.substring(property.lastIndexOf('.') + 1);
    }
}
