/*
 * Copyright (c) 2021, Netherlands Forensic Institute
 * All rights reserved.
 */
package org.hansken.plugin.extraction.test.base;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.startsWith;
import static org.hamcrest.Matchers.stringContainsInOrder;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;

import nl.minvenj.nfi.flits.api.result.ThrowableResult;
import nl.minvenj.nfi.flits.base.DefaultResultValidator;

/**
 * This class is a representation of the FLITS exception handling JSON expected result file.
 * <p>
 * This version expands on the FLITS class by adding 2 new fields:
 * <ul>
 * <li>message.startsWith
 * <li>message.containsInOrder
 * </ul>
 */
final class PluginThrowableResult {
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper().configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);

    private final String _class;
    private final String _message;
    private final String _messageStartsWith;
    private final String[] _messageContainsInOrder;

    private PluginThrowableResult(@JsonProperty("class") final String aClass,
                                  @JsonProperty("message") final String message,
                                  @JsonProperty("message.startsWith") final String messageStartsWith,
                                  @JsonProperty("message.containsInOrder") final String[] messageContainsInOrder) {
        _class = aClass;
        _message = message;
        _messageStartsWith = messageStartsWith;
        _messageContainsInOrder = messageContainsInOrder;
    }

    /**
     * All fields except {@code class} are optional. Multiple or all fields may be present in the JSON, but when
     * {@code message} is present, the validation returns {@code false} assuming that FLITS will continue to validate.
     *
     * @param result the error produced by the plugin
     * @return {@code false} if {@code message} is present, and {@code true} otherwise, <b>assuming no assertions fail</b>.
     */
    private boolean isValidUsingCustomMatching(final ThrowableResult result) {
        final String resultClassName = result.throwable().getClass().getName();
        final String resultMessage = result.throwable().getMessage();

        assertEquals(resultClassName, _class);
        if (_messageStartsWith != null) {
            assertThat(resultMessage, startsWith(_messageStartsWith));
        }
        if (_messageContainsInOrder != null) {
            assertThat(resultMessage, stringContainsInOrder(_messageContainsInOrder));
        }

        // if _message is provided, then no custom matching is necessary, and the ordinary matcher provided by FLITS should validate.
        if (_message != null) {
            return false;
        }
        return true;
    }

    /**
     * This method overrides the validation done by {@link DefaultResultValidator#validate(java.lang.String, java.nio.file.Path)}, by allowing partial matching on exception messages.
     *
     * @param result the error produced by the plugin
     * @param inputPath the expected result file written by the developer to be matched against the {@code result}
     * @return {@code true} if the given {@code inputPath} matches the error produced in {@code result}, {@code false} if {@code message} is present in {@link inputPath}
     * @throws IOException produced by {@link Files#readString(java.nio.file.Path)} or {@link ObjectMapper#readValue(java.lang.String, java.lang.Class)}
     */
    static boolean isValidUsingCustomMatching(final ThrowableResult result, final Path inputPath) throws IOException {
        assertTrue(Files.isRegularFile(inputPath), "No result file found, expected to be located at: " + inputPath);
        final String expectedJson = Files.readString(inputPath);
        final PluginThrowableResult pluginThrowableResult = OBJECT_MAPPER.readValue(expectedJson, PluginThrowableResult.class);
        return pluginThrowableResult.isValidUsingCustomMatching(result);
    }
}
