package org.hansken.plugin.extraction.test.serialize;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

import org.hansken.plugin.extraction.api.RandomAccessData;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientDataContext;
import org.hansken.plugin.extraction.test.base.ProcessingUtil;

import static java.nio.file.Files.exists;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toMap;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.searchTraceFromPath;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.extractFileName;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.getSearchTracesPath;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.isFileWithExtension;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

/**
 * {@link ClientDataContext} implementation which is used by the test framework.
 * This implementation uses a map to store trace data.
 */
public abstract class AbstractTestContext implements ClientDataContext {
    // For each TraceId, this map contains all dataTypes and paths where to find the data
    private final Map<String, Map<String, Path>> _idToDataTypeToPath = new HashMap<>();

    protected AbstractTestContext(final Path tracePath) throws IOException {
        argNotNull("tracePath", tracePath);
        fillData(tracePath);
    }

    /**
     * Store the location of all data provided for search traces.
     *
     * @param tracePath path to the trace
     * @throws IOException when a non existing path is provided
     */
    private void fillData(final Path tracePath) throws IOException {
        final Path searchTracesPath = getSearchTracesPath(tracePath);
        if (!exists(searchTracesPath)) {
            return;
        }

        final Map<Path, String> filenameToId = mapFilenamesToTraceId(searchTracesPath);

        for (Map.Entry<Path, String> entry : filenameToId.entrySet()) {
            final String traceId = entry.getValue();
            final String name = extractFileName(entry.getKey()); // Obtain filename without extension
            final Map<String, Path> dataTypeToPath = mapDataTypeToPath(searchTracesPath, name);
            _idToDataTypeToPath.put(traceId, dataTypeToPath);
        }
    }

    private static Map<String, Path> mapDataTypeToPath(final Path searchTracesPath, final String name) throws IOException {
        try (Stream<Path> files = Files.list(searchTracesPath)) { // List all files
            return files.filter(path -> extractFileName(path).equals(name)) // Keep files corresponding to file path/trace
                .collect(toMap(ProcessingUtil::extractFileExtension, identity()));
        }
    }

    private static Map<Path, String> mapFilenamesToTraceId(final Path searchTracesPath) throws IOException {
        try (Stream<Path> files = Files.list(searchTracesPath)) { // Get all files in this path
            return files.filter(path -> isFileWithExtension(path, "trace")) // Filter to leave only .trace files
                .collect(toMap(Path::getFileName, path -> searchTraceFromPath(path).traceId()));
        }
    }

    @Override
    public RandomAccessData data(final String traceId, final String dataType) {
        argNotNull("traceId", traceId);
        argNotNull("dataType", dataType);
        try {
            return new RandomAccessFileData(_idToDataTypeToPath.get(traceId).get(dataType).toFile());
        } catch (final FileNotFoundException e) {
            throw new IllegalStateException(e);
        }
    }
}
