package org.hansken.plugin.extraction.main;

import static org.hansken.plugin.extraction.test.util.Constants.DEFAULT_VERBOSE_LOGGING_ENABLED;

import static net.sourceforge.argparse4j.impl.Arguments.storeTrue;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import org.junit.platform.engine.TestExecutionResult;
import org.junit.platform.engine.discovery.DiscoverySelectors;
import org.junit.platform.launcher.Launcher;
import org.junit.platform.launcher.LauncherDiscoveryRequest;
import org.junit.platform.launcher.TestExecutionListener;
import org.junit.platform.launcher.TestIdentifier;
import org.junit.platform.launcher.TestPlan;
import org.junit.platform.launcher.core.LauncherDiscoveryRequestBuilder;
import org.junit.platform.launcher.core.LauncherFactory;

import com.google.common.base.Strings;

import net.sourceforge.argparse4j.ArgumentParsers;
import net.sourceforge.argparse4j.inf.ArgumentParser;
import net.sourceforge.argparse4j.inf.ArgumentParserException;
import net.sourceforge.argparse4j.inf.Namespace;

public final class StandaloneTestRunner {

    private static boolean _verboseEnabled = DEFAULT_VERBOSE_LOGGING_ENABLED;

    private StandaloneTestRunner(final Namespace arguments) {
        // initialize the StandAlone Flit test
        final Path testPath = new File(arguments.getString("testpath")).toPath();
        final Path resultPath = new File(arguments.getString("resultpath")).toPath();
        final Boolean regenerate = arguments.getBoolean("regenerate");
        final String host = arguments.getString("host");
        final Integer port = arguments.getInt("port");
        final String pattern = arguments.getString("pattern");
        _verboseEnabled = arguments.getBoolean("verbose");
        StandaloneFlitsTest.setup(testPath, resultPath, regenerate, host, port, pattern);

        final TestReporter testListener = new TestReporter();
        final LauncherDiscoveryRequest request = LauncherDiscoveryRequestBuilder.request()
            .selectors(DiscoverySelectors.selectClass(StandaloneFlitsTest.class))
            .build();
        final Launcher launcher = LauncherFactory.create();
        launcher.discover(request);
        launcher.registerTestExecutionListeners(testListener);
        launcher.execute(request);

        System.exit(testListener.testsFailed() ? 1 : 0);
    }

    public static void main(final String[] args) throws IOException {
        final ArgumentParser parser = argParser();
        Namespace arguments = null;
        try {
            arguments = parser.parseArgs(args);
        }
        catch (final ArgumentParserException e) {
            parser.handleError(e);
            System.exit(1);
        }
        new StandaloneTestRunner(arguments);
    }

    private static ArgumentParser argParser() throws IOException {
        final String description = new String(StandaloneFlitsTest.class.getResourceAsStream("/description.txt").readAllBytes(), StandardCharsets.UTF_8);

        final ArgumentParser parser = ArgumentParsers.newFor("testframework").build()
            .description(description);

        parser.addArgument("-t", "--testpath")
            .metavar("testpath")
            .required(true)
            .help("Absolute or relative path where the test input files are stored");

        parser.addArgument("-r", "--resultpath")
            .metavar("resultpath")
            .required(true)
            .help("Absolute or relative path where the test input files are stored");

        parser.addArgument("-g", "--regenerate")
            .metavar("regenerate")
            .type(Boolean.class)
            .required(false)
            .action(storeTrue())
            .help("Set if the expected test output has to be (re)generated");

        parser.addArgument("-m", "--host")
            .metavar("host")
            .required(true)
            .help("Host where the extraction plugin is served");

        parser.addArgument("-p", "--port")
            .metavar("port")
            .type(Integer.class)
            .required(true)
            .help("Port where the extraction plugin is served");

        parser.addArgument("-i", "--pattern")
            .metavar("pattern")
            .required(false)
            .help(
                "Specify a pattern to run only on specific files from the test path, " +
                    "in the format syntax:pattern (where syntax is filesystem dependant, but " +
                    "at least glob and regex are supported)"
            );

        parser.addArgument("-v", "--verbose")
            .metavar("verbose")
            .type(Boolean.class)
            .required(false)
            .action(storeTrue())
            .help(
                "Enables verbose logging."
            );

        return parser;
    }

    public static boolean isVerboseEnabled() {
        return _verboseEnabled;
    }

    // TODO HANSKEN-14131: issue with FLITS: it only checks existence of the root result path
    // TODO: if one expects a result, but the matcher does not match, you don't notice anything
    // TODO: if the result file path does not exist

    // TODO HANSKEN-14130: issue with this reporter: if it blows up for example due to invalid input path, the test still succeeds

    /**
     * Simple class that listens to the JUnit test runner, prints test information
     * and prints a summary when all tests are finished.
     */
    class TestReporter implements TestExecutionListener {
        private final Deque<String> _path = new ArrayDeque<>();
        private final StringBuilder _output = new StringBuilder();
        private final AtomicInteger _passed = new AtomicInteger();
        private final AtomicInteger _failed = new AtomicInteger();

        @Override
        public void testPlanExecutionFinished(final TestPlan testPlan) {
            System.out.println("");
            System.out.println("Test results:");
            System.out.println(_output.toString());
            System.out.println("  " + _passed.get() + " tests passed / " + _failed.get() + " tests failed");
        }

        @Override
        public void executionStarted(final TestIdentifier testIdentifier) {
            if (testIdentifier.isContainer()) {
                _path.add(testIdentifier.getDisplayName());
            }

            if (testIdentifier.isTest()) {
                final String prefix = _path.stream().skip(3).collect(Collectors.joining("\\"));
                System.out.println("===== Running test: " + prefix + "\\" + testIdentifier.getDisplayName() + " ========");
            }
        }

        @Override
        public void executionFinished(final TestIdentifier testIdentifier, final TestExecutionResult testExecutionResult) {
            if (testIdentifier.isContainer()) {
                _path.removeLast();
            }
            if (testIdentifier.isTest() && testExecutionResult.getStatus() == TestExecutionResult.Status.SUCCESSFUL) {
                logTest(testIdentifier, testExecutionResult);
                _passed.incrementAndGet();
            }
            if (testExecutionResult.getStatus() == TestExecutionResult.Status.FAILED) {
                logTest(testIdentifier, testExecutionResult);
                _failed.incrementAndGet();
                System.out.println(testExecutionResult.getThrowable().get());
                System.out.println("");
            }
            System.out.println();
        }

        private void logTest(final TestIdentifier testIdentifier, final TestExecutionResult testExecutionResult) {
            final String prefix = _path.stream().skip(3).collect(Collectors.joining("\\"));
            _output.append("  ");
            _output.append(Strings.padEnd(prefix + (prefix.isEmpty() ? "" : "\\") + testIdentifier.getDisplayName() + "..", 50, '.'));
            _output.append(testExecutionResult.getStatus().toString());
            _output.append("\n");

            System.out.println("Test " + testExecutionResult.getStatus());
        }

        public boolean testsFailed() {
            return _failed.get() >= 1;
        }
    }
}