package org.hansken.plugin.extraction.test.base;

import static java.util.stream.Collectors.toSet;

import static org.hansken.plugin.extraction.test.util.Constants.DATA_STREAM_NAME_PREFIX;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Set;

import org.hansken.plugin.extraction.api.DataWriter;
import org.hansken.plugin.extraction.api.SearchTrace;
import org.hansken.plugin.extraction.test.ExtractionPluginProcessor;
import org.hansken.plugin.extraction.test.serialize.Deserialize;
import org.hansken.plugin.extraction.test.serialize.TestSearchTrace;
import org.hansken.plugin.extraction.test.util.Constants;

import nl.minvenj.nfi.flits.api.Trace;

/**
 * Utility functions for use when running an {@link ExtractionPluginProcessor}.
 */
public final class ProcessingUtil {

    public static final String SEARCH_TRACES_DIR_NAME = "searchtraces";

    private ProcessingUtil() {
    }

    /**
     * Extract all data types of the {@link org.hansken.plugin.extraction.api.Trace#setData(String, DataWriter) data
     * streams} of given trace. It does this by filtering for all properties which start with the
     * {@link Constants#DATA_STREAM_NAME_PREFIX} marker.
     *
     * @param trace the trace to extract the data types
     * @return all data types of the contained data streams
     */
    public static Set<String> dataTypes(final Trace trace) {
        return trace.properties().stream()
            .filter(property -> property.startsWith(DATA_STREAM_NAME_PREFIX))
            .map(property -> property.substring(DATA_STREAM_NAME_PREFIX.length()))
            .collect(toSet());
    }

    /**
     * Validate that given {@link Trace} contains a data stream of given type.
     *
     * @param trace the trace to test
     * @param dataType the type of the data stream to check for
     * @return {@code true} if the trace contains the requested stream, otherwise {@code false}
     */
    public static boolean hasDataStreamOfType(final Trace trace, final String dataType) {
        return dataStream(trace, dataType) != null;
    }

    /**
     * Extract a {@link org.hansken.plugin.extraction.api.Trace#setData(String, DataWriter) data streams} of given
     * type from given trace.
     *
     * @param trace the trace to read the data stream from
     * @param dataType the type of the data to read
     * @return the data stream as a byte array
     */
    public static byte[] dataStream(final Trace trace, final String dataType) {
        return trace.get(DATA_STREAM_NAME_PREFIX + dataType);
    }

    /**
     * Create a {@link Path} which points to the data stream file of the expected trace and type. The {@code tracePath}
     * is a path which uniquely defines a single (child) trace.
     *
     * @param rootTracePath the path of the root trace JSON
     * @param tracePath a uniquely identifying path for current trace
     * @param dataType the data type of the stream
     * @return a new path for given data stream file
     */
    public static Path createDataStreamPath(final Path rootTracePath, final String tracePath, final String dataType) {
        final String fileName = rootTracePath.getFileName().toString();
        final String fileNameWithoutType = fileName.substring(0, fileName.lastIndexOf('.'));
        return rootTracePath.resolveSibling(fileNameWithoutType + "." + tracePath + "." + dataType);
    }

    /**
     * Create a {@link Path} which points to the output run profile.
     *
     * @param rootTracePath the path of the root trace JSON
     * @return a new path for given output run profile
     */
    public static Path createRunProfilePath(final Path rootTracePath) {
        final String fileName = rootTracePath.getFileName().toString();
        final String fileNameWithoutType = fileName.substring(0, fileName.lastIndexOf('.'));
        return rootTracePath.resolveSibling(fileNameWithoutType + ".profile.txt");
    }

    /**
     * Return filename without extension from a given path.
     *
     * @param path path to a file
     * @return filename without extension
     */
    public static String extractFileName(final Path path) {
        final String fileName = path.getFileName().toString();
        return fileName.substring(0, fileName.lastIndexOf('.'));
    }

    /**
     * Return folder containing search traces given provided path to test trace.
     *
     * @param tracePath path to the trace being tested
     * @return traces returned when querying
     */
    public static Path getSearchTracesPath(final Path tracePath) {
        return Paths.get(String.valueOf(tracePath.getParent()), extractFileName(tracePath), SEARCH_TRACES_DIR_NAME);
    }

    /**
     * Return true if the extension filename located at path equals the provided extension.
     *
     * @param filePath path to file to check
     * @param extension extension to match
     * @return true if matches
     */
    public static boolean isFileWithExtension(final Path filePath, final String extension) {
        final String fileName = filePath.getFileName().toString();
        return extractFileExtension(fileName).equals(extension);
    }

    /**
     * Return extension of file at provided path.
     *
     * @param filePath path to file
     * @return extension of file
     */
    public static String extractFileExtension(final Path filePath) {
        return extractFileExtension(filePath.getFileName().toString());
    }

    private static String extractFileExtension(final String fileName) {
        return fileName.substring(fileName.lastIndexOf(".") + 1);
    }

    /**
     * Create a search trace given the path to the file representing the search trace.
     *
     * @param tracePath Path to the search trace
     * @return New SearchTrace
     */
    public static SearchTrace searchTraceFromPath(final Path tracePath) {
        try {
            return Files.exists(tracePath) ? Deserialize.searchTraceFromJson(tracePath) : new TestSearchTrace("test-input-trace");
        }
        catch (final IOException e) {
            throw new RuntimeException(e);
        }
    }
}
