/*
 * Copyright (c) 2021, Netherlands Forensic Institute
 * All rights reserved.
 */
package nl.minvenj.nfi.flits.serialize;

import java.util.HashMap;
import java.util.Map;

final class MapUtil {

    private static final String ROOT_ELEMENT = "ROOT_ELEMENT.";

    private MapUtil() {
    }

    /**
     * Creates a tree of the provided properties, each with its own value.
     *
     * @param input the properties to process
     * @return tree of properties
     */
    static Map<String, Object> mapMap(final Map<Property, Object> input) {
        final Map<String, Object> result = new HashMap<>();
        for (final Map.Entry<Property, Object> entry : input.entrySet()) {
            final Property property = entry.getKey();
            final Object value = entry.getValue();
            final String rootKey = ROOT_ELEMENT + property.type();
            final Map<String, Object> nested = buildFullNestedMap(property.name(), value);
            mergeInto(result, rootKey, nested);
        }
        return result;
    }

    private static Map<String, Object> buildFullNestedMap(final String processPropertyName, final Object value) {
        if (processPropertyName == null || processPropertyName.isEmpty()) {
            return new HashMap<>();
        }

        final int dotIndex = processPropertyName.indexOf('.');
        if (dotIndex < 0) {
            final Map<String, Object> leafMap = new HashMap<>();
            leafMap.put(processPropertyName, value);
            return leafMap;
        }
        final String propertyType = processPropertyName.substring(0, dotIndex);
        final String propertyName = processPropertyName.substring(dotIndex + 1);
        final Map<String, Object> tailMap = buildFullNestedMap(propertyName, value);

        final Map<String, Object> map = new HashMap<>();
        map.put(propertyType, tailMap);
        return map;
    }

    private static void mergeInto(final Map<String, Object> target, final String rootKey, final Map<String, Object> nestedMap) {
        if (!target.containsKey(rootKey)) {
            target.put(rootKey, new HashMap<>(nestedMap));
            return;
        }

        final Object existing = target.get(rootKey);
        if (existing instanceof Map) {
            mergeNestedMaps((Map<String, Object>) existing, nestedMap);
        }
        else {
            target.put(rootKey, new HashMap<>(nestedMap));
        }
    }

    private static void mergeNestedMaps(final Map<String, Object> target, final Map<String, Object> source) {
        for (final Map.Entry<String, Object> entry : source.entrySet()) {
            final String key = entry.getKey();
            final Object sourceValue = entry.getValue();

            if (target.containsKey(key)) {
                final Object targetValue = target.get(key);

                if (targetValue instanceof Map && sourceValue instanceof Map) {
                    mergeNestedMaps((Map<String, Object>) targetValue, (Map<String, Object>) sourceValue);
                }
                else if (targetValue instanceof Map) {
                    // Target is map, source is single value, keep target map
                    // Quietly ignore, this case can not be solved
                }
                else {
                    // One of:
                    // 1. Target single value, source map, replace
                    // 2. Both values, overwrite
                    target.put(key, sourceValue);
                }
            }
            else {
                target.put(key, sourceValue);
            }
        }
    }

    static String getMapNamePostfix(final Property property) {
        final int beginIndex = property.fullName().indexOf(".");
        return property.fullName().substring(beginIndex + 1);
    }
}
