package org.hansken.plugin.extraction.test;

import java.nio.file.Path;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Optional;

import org.hansken.plugin.extraction.api.DeferredExtractionPlugin;
import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.LatLong;
import org.hansken.plugin.extraction.api.MetaExtractionPlugin;
import org.hansken.plugin.extraction.api.SearchTrace;
import org.hansken.plugin.extraction.api.Trace;
import org.hansken.plugin.extraction.api.TraceSearcher;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientTrace;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientDataContext;
import org.hansken.plugin.extraction.test.base.DefaultExtractionPluginProcessor;
import org.hansken.plugin.extraction.test.base.DefaultPluginResultGenerator;
import org.hansken.plugin.extraction.test.base.DefaultPluginResultValidator;

import nl.minvenj.nfi.flits.api.Flits;
import nl.minvenj.nfi.flits.api.FlitsProcessor;
import nl.minvenj.nfi.flits.api.FlitsResult;
import nl.minvenj.nfi.flits.api.FlitsResultGenerator;
import nl.minvenj.nfi.flits.api.FlitsResultValidator;
import nl.minvenj.nfi.flits.api.result.EmptyResult;
import nl.minvenj.nfi.flits.api.result.ThrowableResult;
import nl.minvenj.nfi.flits.api.result.TraceResult;
import nl.minvenj.nfi.flits.serialize.TraceToJson;

/**
 * Base implementation of the {@link Flits} framework {@link FlitsProcessor processor}, for use with
 * {@link ExtractionPlugin external extraction plugins}.
 * <p>
 * It is up to the implementation to decide how to translate a test input file to
 * a test case, see for example {@link DefaultExtractionPluginProcessor}
 *
 * @see EmbeddedExtractionPluginFlits
 */
public interface ExtractionPluginProcessor extends FlitsProcessor {

    TraceToJson TRACE_TO_JSON = TraceToJson.create()
        .addCustomConversion(byte[].class, Arrays::toString)
        .addCustomConversion(ZonedDateTime.class, ZonedDateTime::toString)
        .addCustomConversion(LatLong.class, LatLong::toISO6709);

    @Override
    default FlitsResultGenerator generator() {
        return new DefaultPluginResultGenerator(TRACE_TO_JSON);
    }

    @Override
    default FlitsResultValidator validator() {
        return new DefaultPluginResultValidator(TRACE_TO_JSON);
    }

    /**
     * Process the given input {@link Trace trace} (which may be empty) and {@link ClientDataContext context}
     * taken from the input path given to the base {@link #process(Path, Path)} method, using a certain
     * {@link ExtractionPlugin plugin}.
     * <p>
     * In case of a {@link MetaExtractionPlugin meta} extraction, the context will have datatype {@code 'meta'}
     * and no data stream.
     * <p>
     * The given {@link FlitsResult result} should contain and describe the outcome. There are
     * three kinds of possible results:
     * <ul>
     *     <li>{@link EmptyResult}: the matcher did not match on the input</li>
     *     <li>{@link ThrowableResult}: running the plugin resulted in an exception or error</li>
     *     <li>{@link TraceResult}: running the plugin resulted in (new) trace information</li>
     * </ul>
     *
     * @param trace the trace to process
     * @param context context of this trace extraction
     * @return the result of processing the given input
     */
    Optional<FlitsResult> process(ClientTrace trace, ClientDataContext context);

    /**
     * Process the given input {@link Trace trace} (which may be empty) and {@link ClientDataContext context}
     * taken from the input path given to the base {@link #process(Path, Path)} method, using a certain
     * {@link DeferredExtractionPlugin plugin}. {@link TraceSearcher searcher} will apply its search method
     * to all traces available from the input path.
     * <p>
     * The given {@link FlitsResult result} should contain and describe the outcome. There are
     * three kinds of possible results:
     * <ul>
     *     <li>{@link EmptyResult}: the matcher did not match on the input</li>
     *     <li>{@link ThrowableResult}: running the plugin resulted in an exception or error</li>
     *     <li>{@link TraceResult}: running the plugin resulted in (new) trace information</li>
     * </ul>
     *
     * @param trace the trace to process
     * @param context context of this trace extraction
     * @param searcher searcher to use for searching additional {@link SearchTrace} traces
     * @return the result of processing the given input
     */
    Optional<FlitsResult> processDeferred(ClientTrace trace, ClientDataContext context, TraceSearcher searcher);
}
