package org.hansken.plugin.extraction.test.base;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.stream.Collectors.toList;

import static org.hansken.plugin.extraction.test.base.ProcessingUtil.createDataStreamPath;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.createRunProfilePath;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.dataStream;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.dataTypes;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.hansken.plugin.extraction.api.DataWriter;
import org.hansken.plugin.extraction.runtime.grpc.client.PluginRunProfile;
import org.hansken.plugin.extraction.test.base.TraceDelegate.EmbeddedTraceDelegate;

import nl.minvenj.nfi.flits.api.Trace;
import nl.minvenj.nfi.flits.api.result.TraceResult;
import nl.minvenj.nfi.flits.base.DefaultResultGenerator;
import nl.minvenj.nfi.flits.serialize.TraceToJson;

/**
 * Extension to the {@link DefaultPluginResultGenerator} which also writes each data stream of a
 * {@link org.hansken.plugin.extraction.api.Trace} to a separate file (see
 * {@link org.hansken.plugin.extraction.api.Trace#setData(String, DataWriter)}).
 */
public class DefaultPluginResultGenerator extends DefaultResultGenerator {

    public DefaultPluginResultGenerator(final TraceToJson traceToJson) {
        super(traceToJson);
    }

    @Override
    public void generate(final TraceResult result, final Path outputPath) throws IOException {
        super.generate(result, outputPath);
        writeDataStreams(outputPath, result.traces());
        writeProfile(outputPath, result.traces());
    }

    private void createOutputFolder(final Path path) {
        try {
            if (!Files.exists(path.getParent())) {
                // accept the possible race condition
                Files.createDirectories(path.getParent());
            }
        }
        catch (final IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void writeDataStreams(final Path outputPath, final List<Trace> traces) throws IOException {
        for (final Trace trace : traces) {
            for (final String dataType : dataTypes(trace)) {
                final Path dataFile = createDataStreamPath(outputPath, trace.get("id"), dataType);
                createOutputFolder(dataFile);
                Files.write(dataFile, dataStream(trace, dataType));
            }

            writeDataStreams(outputPath, trace.children().collect(toList()));
        }
    }

    private void writeProfile(final Path outputPath, final List<Trace> traces) {
        traces.stream()
            .filter(trace -> trace instanceof EmbeddedTraceDelegate) // could be an plain (embedded) Trace which has no profile()
            .map(trace -> (EmbeddedTraceDelegate) trace)
            .filter(trace -> !(trace.profile().isEmpty() ||
                (trace.profile().get().getDoubles()).isEmpty() && trace.profile().get().getIntegers().isEmpty()))
            .peek(trace -> createOutputFolder(createRunProfilePath(outputPath)))
            .forEach(trace -> writeProfileToFile(trace, createRunProfilePath(outputPath).toFile()));
    }

    private void writeProfileToFile(final EmbeddedTraceDelegate trace, final File outputFile) {
        try {
            try (BufferedWriter writer = new BufferedWriter(new FileWriter(outputFile, UTF_8))) {
                writer.write("# this file contains all recorded run profiles for processing " + trace.name() + ".\n");
                writer.write("# this information can be useful when optimizing the plugins performance.\n");

                for (final String line : profileLines(trace.profile().get())) {
                    writer.write(line);
                    writer.newLine();
                }
            }
        }
        catch (final IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private List<String> profileLines(final PluginRunProfile profile) {
        final List<String> profileOutput = new ArrayList<>();
        profile.getIntegers().forEach((key, value) -> {
            profileOutput.add(key + ": " + value);
        });
        profile.getDoubles().forEach((key, value) -> {
            profileOutput.add(key + ": " + value);
        });
        Collections.sort(profileOutput);
        return profileOutput;
    }
}
