package org.hansken.plugin.extraction.test.serialize;

import static org.hansken.plugin.extraction.runtime.grpc.common.VersionUtil.getApiVersion;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.getSearchTracesPath;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.isFileWithExtension;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.stream.Stream;

import org.hansken.plugin.extraction.api.BatchSearchResult;
import org.hansken.plugin.extraction.api.SearchOptions;
import org.hansken.plugin.extraction.api.SearchResult;
import org.hansken.plugin.extraction.api.SearchScope;
import org.hansken.plugin.extraction.api.SearchSortOption;
import org.hansken.plugin.extraction.api.SearchTrace;
import org.hansken.plugin.extraction.api.TraceSearcher;
import org.hansken.plugin.extraction.hql_lite.lang.human.HqlLiteHumanQueryParser;
import org.hansken.plugin.extraction.hql_lite.lang.human.matcher.HqlMatcher;
import org.hansken.plugin.extraction.test.base.ProcessingUtil;

public class TestTraceSearcher implements TraceSearcher {

    private final Path _tracePath;

    public TestTraceSearcher(final Path tracePath) {
        _tracePath = tracePath;
    }

    @Override
    public SearchResult search(final String query, final int count) throws InterruptedException, ExecutionException {
        return search(query, count, SearchOptions.DEFAULT);
    }

    @Override
    public SearchResult search(final String query, final int count, final SearchOptions options) throws InterruptedException, ExecutionException {
        return search(query, count, options.scope(), options.start(), options.sort());
    }

    @Override
    public SearchResult search(final String query, final int count, final SearchScope scope, final int start, final List<SearchSortOption> sort) throws InterruptedException, ExecutionException {
        // Scope is ignored for the testTraceSearcher, as we are not actually extracting images

        final SearchTrace[] matchingTraces = getAllSearchTraces(HqlLiteHumanQueryParser.parse(query, getApiVersion()));

        final BatchSearchResult result = new BatchSearchResult(matchingTraces.length);
        result.setTraces(matchingTraces);

        return result;
    }

    private SearchTrace[] getAllSearchTraces(final HqlMatcher matcher) throws ExecutionException {
        try (Stream<Path> files = Files.list(getSearchTracesPath(_tracePath))) { // Get all files in this path
            return files.filter(path -> isFileWithExtension(path, "trace")) // Get .trace files only
                .sorted() //There is no guarantee that the result of Files.list() will appear in any specific order
                .map(ProcessingUtil::searchTraceFromPath) // Create SearchTrace object from .trace files
                .filter(searchTrace -> searchTrace.getDataTypes().stream() // For each searchTrace get data types
                    .anyMatch(dataType -> matcher.match(searchTrace, dataType))) // Check for trace matching matcher
                .toArray(SearchTrace[]::new); // Convert the trace files to SearchTrace
        }
        catch (final IOException e) {
            throw new ExecutionException(e);
        }
    }
}
