package org.hansken.plugin.extraction.test.base;

import static org.hansken.plugin.extraction.runtime.grpc.common.VersionUtil.getApiVersion;
import static org.hansken.plugin.extraction.test.base.ProcessingUtil.SEARCH_TRACES_DIR_NAME;
import static org.hansken.plugin.extraction.test.serialize.Deserialize.contextFromRandomAccessData;
import static org.hansken.plugin.extraction.test.serialize.Deserialize.traceFromJson;
import static org.hansken.plugin.extraction.test.serialize.Deserialize.traceSearcherFromData;
import static org.hansken.plugin.extraction.test.util.HqlLogger.HQL_MATCH_MESSAGE;
import static org.hansken.plugin.extraction.test.util.HqlLogger.HQL_NO_MATCH_MESSAGE;
import static org.hansken.plugin.extraction.test.util.TestResourceUtil.relativePathToWorkingDir;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;

import org.hansken.plugin.extraction.api.BaseExtractionPlugin;
import org.hansken.plugin.extraction.api.DataContext;
import org.hansken.plugin.extraction.api.DeferredExtractionPlugin;
import org.hansken.plugin.extraction.api.ExtractionPlugin;
import org.hansken.plugin.extraction.api.PluginInfo;
import org.hansken.plugin.extraction.api.PluginType;
import org.hansken.plugin.extraction.api.TraceSearcher;
import org.hansken.plugin.extraction.hql_lite.lang.human.HqlLiteHumanQueryParser;
import org.hansken.plugin.extraction.runtime.grpc.client.ExtractionPluginClient;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientDataContext;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientTrace;
import org.hansken.plugin.extraction.runtime.grpc.common.Pack;
import org.hansken.plugin.extraction.runtime.grpc.common.Unpack;
import org.hansken.plugin.extraction.test.ExtractionPluginProcessor;
import org.hansken.plugin.extraction.test.TestFrameworkException;
import org.hansken.plugin.extraction.test.serialize.MetaTestContext;
import org.hansken.plugin.extraction.test.serialize.TestTrace;
import org.hansken.plugin.extraction.test.util.HqlLogger;
import org.opentest4j.AssertionFailedError;

import nl.minvenj.nfi.flits.api.FlitsResult;
import nl.minvenj.nfi.flits.api.FlitsResultValidator;
import nl.minvenj.nfi.flits.api.Trace;
import nl.minvenj.nfi.flits.api.result.ThrowableResult;
import nl.minvenj.nfi.flits.api.result.TraceResult;

/**
 * Base implementation of an {@link ExtractionPluginProcessor}, which simply delegates
 * the {@link ExtractionPluginProcessor#process(ClientTrace, ClientDataContext)} call to the given {@link ExtractionPlugin}.
 * <p>
 * Depending on the {@link PluginType type} of plugin under test, the test case input is used as follows:
 * <ul>
 *     <li>
 *         {@link PluginType#EXTRACTION_PLUGIN} or {@link PluginType#DEFERRED_EXTRACTION_PLUGIN}: the implementation
 *         loads the file pointed to by the given input path as the data stream. Optionally, a sibling file may be
 *         present with extension {@code .trace}, which will
 *         be taken as the base input {@link Trace}.
 *     </li>
 *     <li>
 *         {@link PluginType#META_EXTRACTION_PLUGIN} or {@link PluginType#DEFERRED_META_EXTRACTION_PLUGIN}: the
 *         implementation deserializes the file pointed to by the given input as the input {@link Trace}.
 *     </li>
 * </ul>
 */
public abstract class DefaultExtractionPluginProcessor implements ExtractionPluginProcessor {

    protected final PluginInfo _info;

    /**
     * Create a new default processor which delegates to given {@link ExtractionPlugin}.
     *
     * @param plugin the plugin to run the test with
     */
    private DefaultExtractionPluginProcessor(final BaseExtractionPlugin plugin) {
        _info = Unpack.pluginInfo(Pack.pluginInfo(argNotNull("plugin", plugin).pluginInfo()));
    }

    @Override
    public String name() {
        if (_info.id() != null) {
            return _info.id().name();
        }
        return _info.name();
    }

    @Override
    public Optional<FlitsResult> process(final Path inputPath, final Path relativePath) throws IOException {
        try {
            final String fileName = inputPath.getFileName().toString();
            return switch (_info.pluginType()) {
                case EXTRACTION_PLUGIN -> processPlugin(inputPath, fileName);
                case DEFERRED_EXTRACTION_PLUGIN -> processDeferredPlugin(inputPath, fileName);
                case META_EXTRACTION_PLUGIN -> processMetaPlugin(inputPath);
                case DEFERRED_META_EXTRACTION_PLUGIN -> processDeferredMetaPlugin(inputPath, fileName);
            };
        }
        catch (final Exception e) {
            System.err.println("The extraction plugin test framework failed running the test for " + relativePathToWorkingDir(inputPath));
            System.err.println();
            if (e instanceof TestFrameworkException) {
                printTestFrameworkException((TestFrameworkException) e);
            }
            else {
                printException(e);
            }
            // mark the test as failed by throwing an assertion exception (thrown by fail()).
            // this eventually prints a non-meaningful stacktrace to the user, so warn the user that they can ignore this message
            System.err.println();
            System.err.println("-------------------------------------------------");
            System.err.println("The following lines can be ignored:");
            throw new AssertionFailedError();
        }
    }

    private static void printException(final Exception exception) {
        System.err.println("We can't give a clear description on why the test failed, but the following information could give some details on the failure:");
        exception.printStackTrace();
    }

    private static void printTestFrameworkException(final TestFrameworkException exception) {
        if (exception.getMessage() != null) {
            System.err.println("Cause of the failure:");
            System.err.println(exception.getMessage());
            System.err.println();
        }
        System.err.println("> Please create a Hansken extraction plugin SDK bug report if you think this failure is incorrect.");
        if (exception.getCause() != null && exception.getCause().getMessage() != null) {
            System.err.println();
            System.err.println("-------------------------------------------------");
            System.err.println("The following lines could give some more details on the failure:");
            System.err.println();
            System.err.println(exception.getCause().getMessage());
        }
    }

    private Optional<FlitsResult> processPlugin(final Path inputPath, final String fileName) throws Exception {
        final String fileNameWithoutType = fileName.substring(0, fileName.lastIndexOf('.'));
        final Path tracePath = inputPath.resolveSibling(fileNameWithoutType + ".trace");
        try (ClientTrace trace = getTrace(tracePath)) {
            final ClientDataContext context = contextFromRandomAccessData(inputPath);
            return process(trace, context);
        }
    }

    private Optional<FlitsResult> processDeferredPlugin(final Path inputPath, final String fileName) throws Exception {
        final String fileNameWithoutType = fileName.substring(0, fileName.lastIndexOf('.'));
        final Path tracePath = inputPath.resolveSibling(fileNameWithoutType + ".trace");
        try (ClientTrace trace = getTrace(tracePath)) {
            final ClientDataContext context = contextFromRandomAccessData(inputPath);
            final TraceSearcher searcher = traceSearcherFromData(inputPath);
            return processDeferred(trace, context, searcher);
        }
    }

    private Optional<FlitsResult> processMetaPlugin(final Path inputPath) throws Exception {
        try (ClientTrace trace = getTrace(inputPath)) {
            return process(trace, new MetaTestContext(inputPath));
        }
    }

    private Optional<FlitsResult> processDeferredMetaPlugin(final Path inputPath, final String fileName) throws Exception {
        final String fileNameWithoutType = fileName.substring(0, fileName.lastIndexOf('.'));
        final Path tracePath = inputPath.resolveSibling(fileNameWithoutType + ".trace");
        try (ClientTrace trace = getTrace(tracePath)) {
            final MetaTestContext context = new MetaTestContext(inputPath);
            final TraceSearcher searcher = traceSearcherFromData(inputPath);
            return processDeferred(trace, context, searcher);
        }
    }

    private ClientTrace getTrace(final Path tracePath) {
        return Files.exists(tracePath) && !isSearchTrace(tracePath) ? traceFromJson(tracePath) : new TestTrace("test-input-trace");
    }

    /**
     * {@link DefaultExtractionPluginProcessor} for embedded extraction plugins.
     *
     * @param plugin the extraction plugin to be used
     * @param verboseLoggingEnabled enable verbose logging for HQL queries
     * @return {@link DefaultExtractionPluginProcessor} for embedded extraction plugins.
     */
    public static DefaultExtractionPluginProcessor embedded(final ExtractionPlugin plugin, final boolean verboseLoggingEnabled) {
        return processor(plugin, new DefaultPluginResultValidator(TRACE_TO_JSON), verboseLoggingEnabled);
    }

    /**
     * {@link DefaultExtractionPluginProcessor} for embedded extraction plugins using a custom validator.
     *
     * @param plugin the extraction plugin to be used
     * @param validator validator for the Flits result
     * @param verboseLoggingEnabled enable verbose logging for HQL queries
     * @return {@link DefaultExtractionPluginProcessor} for embedded extraction plugins.
     */
    public static DefaultExtractionPluginProcessor embedded(final ExtractionPlugin plugin, final FlitsResultValidator validator, final boolean verboseLoggingEnabled) {
        return processor(plugin, validator, verboseLoggingEnabled);
    }

    /**
     * {@link DefaultExtractionPluginProcessor} for embedded deferred extraction plugins.
     *
     * @param plugin the deferred extraction plugin to be used
     * @param verboseLoggingEnabled enable verbose logging for HQL queries
     * @return {@link DefaultExtractionPluginProcessor} for embedded deferred extraction plugins.
     */
    public static DefaultExtractionPluginProcessor embedded(final DeferredExtractionPlugin plugin, final boolean verboseLoggingEnabled) {
        return processor(plugin, new DefaultPluginResultValidator(TRACE_TO_JSON), verboseLoggingEnabled);
    }

    /**
     * {@link DefaultExtractionPluginProcessor} for embedded deferred extraction plugins using a custom validator.
     *
     * @param plugin the deferred extraction plugin to be used
     * @param validator validator for the Flits result
     * @param verboseLoggingEnabled enable verbose logging for HQL queries
     * @return {@link DefaultExtractionPluginProcessor} for embedded deferred extraction plugins.
     */
    public static DefaultExtractionPluginProcessor embedded(final DeferredExtractionPlugin plugin, final FlitsResultValidator validator, final boolean verboseLoggingEnabled) {
        return processor(plugin, validator, verboseLoggingEnabled);
    }

    private static DefaultExtractionPluginProcessor processor(final BaseExtractionPlugin plugin, final FlitsResultValidator validator, final boolean verboseLoggingEnabled) {
        return new DefaultExtractionPluginProcessor(plugin) {
            @Override
            @SuppressWarnings("checkstyle:illegalcatch")
            public Optional<FlitsResult> process(final ClientTrace trace, final ClientDataContext context) {
                try {
                    if (!isHqlMatch(trace, context, verboseLoggingEnabled)) {
                        return Optional.empty();
                    }
                    final TraceDelegate.EmbeddedTraceDelegate delegate = TraceDelegate.forEmbeddedTrace(trace);
                    if (!(plugin instanceof final ExtractionPlugin extractionPlugin)) {
                        throw new IllegalArgumentException("Provided plugin is not an implementation of ExtractionPlugin.");
                    }
                    extractionPlugin.process(delegate, context);

                    return Optional.of(new TraceResult(delegate));
                }
                catch (final Throwable t) {
                    return Optional.of(new ThrowableResult(t));
                }
            }

            @Override
            @SuppressWarnings("checkstyle:illegalcatch")
            public Optional<FlitsResult> processDeferred(final ClientTrace trace, final ClientDataContext context, final TraceSearcher searcher) {
                try {
                    if (!isHqlMatch(trace, context, verboseLoggingEnabled)) {
                        return Optional.empty();
                    }
                    final TraceDelegate.EmbeddedTraceDelegate delegate = TraceDelegate.forEmbeddedTrace(trace);
                    if (!(plugin instanceof final DeferredExtractionPlugin deferred)) {
                        throw new IllegalArgumentException("Provided plugin is not an implementation of DeferredExtractionPlugin.");
                    }
                    deferred.process(delegate, context, searcher);

                    return Optional.of(new TraceResult(delegate));
                }
                catch (final InterruptedException e) {
                    Thread.currentThread().interrupt();
                    return Optional.of(new ThrowableResult(e));
                }
                catch (final Throwable t) {
                    return Optional.of(new ThrowableResult(t));
                }
            }

            @Override
            public FlitsResultValidator validator() {
                return validator;
            }
        };
    }

    /**
     * {@link DefaultExtractionPluginProcessor} for remote extraction plugins.
     *
     * @param client client that is connects to the remote extraction plugin
     * @param verboseLoggingEnabled enable verbose logging for HQL queries
     * @return {@link DefaultExtractionPluginProcessor} for remote extraction plugins.
     */
    public static DefaultExtractionPluginProcessor remote(final ExtractionPluginClient client, final boolean verboseLoggingEnabled) {
        return new DefaultExtractionPluginProcessor(client) {
            @Override
            @SuppressWarnings("checkstyle:illegalcatch")
            public Optional<FlitsResult> process(final ClientTrace trace, final ClientDataContext context) {
                try {
                    if (!isHqlMatch(trace, context, verboseLoggingEnabled)) {
                        return Optional.empty();
                    }
                    final TraceDelegate.ClientTraceDelegate delegate = TraceDelegate.forClientTrace(trace);
                    client.process(delegate, context);

                    return Optional.of(new TraceResult(delegate.getEmbeddedDelegate()));
                }
                catch (final Throwable t) {
                    return Optional.of(new ThrowableResult(t));
                }
            }

            @Override
            @SuppressWarnings("checkstyle:illegalcatch")
            public Optional<FlitsResult> processDeferred(final ClientTrace trace, final ClientDataContext context, final TraceSearcher searcher) {
                try {
                    if (!isHqlMatch(trace, context, verboseLoggingEnabled)) {
                        return Optional.empty();
                    }
                    final TraceDelegate.ClientTraceDelegate delegate = TraceDelegate.forClientTrace(trace);
                    client.processDeferred(delegate, context, searcher);

                    return Optional.of(new TraceResult(delegate.getEmbeddedDelegate()));

                }
                catch (final Throwable t) {
                    return Optional.of(new ThrowableResult(t));
                }
            }
        };
    }

    protected boolean isHqlMatch(final ClientTrace trace, final DataContext context, final boolean verboseLoggingEnabled) {
        final String hqlMatcher = _info.hqlMatcher();
        final String dataType = context.dataType();
        if (!HqlLiteHumanQueryParser.parse(_info.hqlMatcher(), getApiVersion()).match(trace, context.dataType())) {
            HqlLogger.logTrace(trace, dataType, hqlMatcher, HQL_NO_MATCH_MESSAGE, verboseLoggingEnabled);
            return false;
        }
        HqlLogger.logTrace(trace, dataType, hqlMatcher, HQL_MATCH_MESSAGE, verboseLoggingEnabled);

        return true;
    }

    // default visibility for access in test
    static boolean isSearchTrace(final Path tracePath) {
        final String parentFolderName = tracePath.getParent().getFileName().toString();
        return parentFolderName.equals(SEARCH_TRACES_DIR_NAME);
    }
}
