package org.hansken.plugin.extraction.test.base;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import org.hansken.plugin.extraction.api.DataWriter;
import org.hansken.plugin.extraction.api.transformations.DataTransformation;
import org.hansken.plugin.extraction.runtime.grpc.client.PluginRunProfile;
import org.hansken.plugin.extraction.runtime.grpc.client.api.ClientTrace;
import org.hansken.plugin.extraction.util.ThrowingConsumer;

import nl.minvenj.nfi.flits.api.Trace;

/**
 * Flits {@link Trace} and extraction plugin trace implementations for embedded and remote traces.
 * Two types of Trace Delegates can be instantiated: one for embedded extraction plugins, and one for remote extraction plugins.
 * <p>
 * These implementations capture all stored child traces so they can be exposed using {@link Trace#children()}, for use with the Flits test framework.
 */
final class TraceDelegate {

    private TraceDelegate() {
        // hide utility constructor
    }

    /**
     * Create an interceptor for the given extraction plugin {@link org.hansken.plugin.extraction.api.Trace}.
     * Used for embedded extraction plugins.
     *
     * @param trace the trace to wrap and intercept calls from
     * @return the wrapped trace
     */
    public static EmbeddedTraceDelegate forEmbeddedTrace(final org.hansken.plugin.extraction.api.Trace trace) {
        return new EmbeddedTraceDelegate(trace);
    }

    /**
     * Create an interceptor for the given extraction plugin {@link ClientTrace}.
     * Used for remote (gRPC) extraction plugins.
     *
     * @param trace the trace to wrap and intercept calls from
     * @return the wrapped trace
     */
    public static ClientTraceDelegate forClientTrace(final ClientTrace trace) {
        return new ClientTraceDelegate(trace);
    }

    /**
     * Flits {@link Trace} and extraction plugin {@link org.hansken.plugin.extraction.api.Trace}
     * implementation which captures all stored child traces so they can be exposed using {@link #children()},
     * for use with the Flits test framework.
     */
    static class EmbeddedTraceDelegate implements Trace, org.hansken.plugin.extraction.api.Trace {
        protected final List<Trace> _children;
        private final org.hansken.plugin.extraction.api.Trace _delegate;

        EmbeddedTraceDelegate(final org.hansken.plugin.extraction.api.Trace trace) {
            _delegate = argNotNull("trace", trace);
            _children = new ArrayList<>();
        }

        @Override
        public String traceId() {
            return _delegate.traceId();
        }

        @Override
        public Set<String> types() {
            return _delegate.types();
        }

        @Override
        public org.hansken.plugin.extraction.api.Trace addType(final String s) {
            _delegate.addType(s);
            return this;
        }

        @Override
        public Set<String> properties() {
            // The Hansken trace model has map properties, however, on the Java API it is not possible
            // to pass a Java 'Map' as the value of said property instead, each map value is set using
            // an expanded key of the format `type.property.key` (e.g. set("file.timestamps.addedOn", timestamp).

            // The libraries, and with it FLITS, handle map properties the other way around.
            // This method and the one below mimic the Hansken style map properties as being library style.
            // TODO HANSKEN-14108: when FLITS support is integrated in Hansken, this should be resolved
            return _delegate.properties();
        }

        @Override
        public <T> T get(final String property) {
            return _delegate.get(property);
        }

        @Override
        public org.hansken.plugin.extraction.api.Trace set(final String property, final Object value) {
            _delegate.set(property, value);
            return this;
        }

        @Override
        public org.hansken.plugin.extraction.api.Trace addTracelet(final Tracelet tracelet) {
            _delegate.addTracelet(tracelet);
            return this;
        }

        @Override
        public List<Tracelet> getNewTracelets() {
            return _delegate.getNewTracelets();
        }

        @Override
        public org.hansken.plugin.extraction.api.Trace setData(final String dataType, final DataWriter writer) throws IOException {
            _delegate.setData(dataType, writer);
            return this;
        }

        @Override
        public org.hansken.plugin.extraction.api.Trace setData(final String dataType, final List<DataTransformation> transformations) {
            _delegate.setData(dataType, transformations);
            return this;
        }

        @Override
        public org.hansken.plugin.extraction.api.Trace newChild(final String name, final ThrowingConsumer<org.hansken.plugin.extraction.api.Trace, IOException> enrichChildCallback) throws IOException {
            _delegate.newChild(name, child -> {
                final EmbeddedTraceDelegate delegateChild = TraceDelegate.forEmbeddedTrace(child);
                enrichChildCallback.accept(delegateChild);
                _children.add(delegateChild);
            });
            return this;
        }

        @Override
        public Stream<Trace> children() {
            return _children.stream();
        }
    }

    /**
     * Flits {@link Trace} and extraction plugin {@link ClientTrace} implementation which captures all
     * stored child traces so they can be exposed using {@link #children()}, for use with the Flits test framework.
     * <p>
     * {@link ClientTrace} implementation follows the {@link EmbeddedTraceDelegate} implementation, but overrides the create-child
     * functionality since children should be handled differently for each type.
     */
    static class ClientTraceDelegate extends EmbeddedTraceDelegate implements ClientTrace {

        private final ClientTraceDelegate _parent;
        private final ClientTrace _delegate;

        ClientTraceDelegate(final ClientTrace trace) {
            this(null, trace);
        }

        ClientTraceDelegate(final ClientTraceDelegate parent, final ClientTrace trace) {
            super(argNotNull("trace", trace));
            _delegate = trace;
            _parent = parent;
        }

        @Override
        public org.hansken.plugin.extraction.api.Trace newChild(final String name, final ThrowingConsumer<org.hansken.plugin.extraction.api.Trace, IOException> enrichChildCallback) {
            throw new UnsupportedOperationException("implementation of a client trace does not support creating children by using a callback");
        }

        @Override
        public String name() {
            return _delegate.name();
        }

        @Override
        public ClientTrace newChild(final String name) {
            return new ClientTraceDelegate(this, _delegate.newChild(name));
        }

        @Override
        public void save() {
            if (_parent != null) {
                _parent._children.add(this);
            }
        }

        @Override
        public void close() throws Exception {
            ClientTrace.super.close();
        }

        @Override
        public void profile(final PluginRunProfile profile) {
            _delegate.profile(profile);
        }

        @Override
        public Optional<PluginRunProfile> profile() {
            return _delegate.profile();
        }
    }
}
