package org.hansken.plugin.extraction.test;

import org.hansken.plugin.extraction.api.PluginType;

import nl.minvenj.nfi.flits.api.Flits;
import static org.hansken.plugin.extraction.test.util.Constants.DEFAULT_VERBOSE_LOGGING_ENABLED;

/**
 * Base class for Extraction Plugin tests using the {@link Flits} framework.
 * This class defines input and result file patterns used in the implementations
 * {@link EmbeddedExtractionPluginFlits} and {@link RemoteExtractionPluginFlits}.
 * <p>
 * Depending on the {@link PluginType type} of plugin under test, the test input is selected as follows:
 * <ul>
 *     <li>
 *         {@link PluginType#EXTRACTION_PLUGIN} or {@link PluginType#DEFERRED_EXTRACTION_PLUGIN}: all files matching
 *         {@link #DATA_STREAM_INPUT_PATTERN}
 *     </li>
 *     <li>
 *         {@link PluginType#META_EXTRACTION_PLUGIN}: all files matching {@link #TRACE_INPUT_PATTERN}
 *     </li>
 * </ul>
 */
public abstract class ExtractionPluginFlits extends Flits {

    // every file not ending with .trace or .flits.gz suffix is taken as data stream input
    private static final String DATA_STREAM_INPUT_PATTERN = "regex:.*(?<!\\.trace|\\.flits\\.gz)$";
    // every file ending with .trace is taken as trace input
    private static final String TRACE_INPUT_PATTERN = "regex:.*\\.trace$";

    protected abstract PluginType pluginType();

    @Override
    public String pattern() {
        switch (pluginType()) {
            case EXTRACTION_PLUGIN:
            case DEFERRED_EXTRACTION_PLUGIN:
                return DATA_STREAM_INPUT_PATTERN;
            case META_EXTRACTION_PLUGIN:
            case DEFERRED_META_EXTRACTION_PLUGIN:
                return TRACE_INPUT_PATTERN;
            default:
                throw new AssertionError("unknown type of plugin: " + pluginType());
        }
    }

    @Override
    public String resultFileSuffix() {
        return "." + processor().name() + ".trace";
    }

    /**
     * Enable verbose logging for HQL queries.
     * {@link org.hansken.plugin.extraction.test.util.HqlLogger} class will give more verbose logging output when verbose logging is enabled.
     *
     * @return verbose logging enabled
     */
    protected boolean isVerboseLoggingEnabled() {
        return DEFAULT_VERBOSE_LOGGING_ENABLED;
    }
}
