package org.hglteam.platform.config.settings;

import org.hglteam.platform.settings.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.core.io.Resource;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.time.ZoneId;
import java.util.Collections;
import java.util.Optional;
import java.util.function.Predicate;

public final class SettingsFactory {
    private static final Logger log = LoggerFactory.getLogger(SettingsFactory.class);

    private SettingsFactory() { }

    public static final String APPLICATION_SECURITY_AUTHORIZATIONS = "application.security.authorizations";
    public static final String APPLICATION_SECURITY_CLAIM_PATHS_EMAIL = "application.security.claim.paths.email";
    public static final String APPLICATION_SECURITY_CLAIM_PATHS_PREFERRED_USERNAME = "application.security.claim.paths.preferred-username";
    public static final String APPLICATION_SECURITY_CLAIM_PATHS_USER_IDENTIFIER = "application.security.claim.paths.user-identifier";
    public static final String APPLICATION_SECURITY_CLAIM_PATHS_REALM_ACCESS = "application.security.claim.paths.realm-access";
    public static final String APPLICATION_SECURITY_CORS_MAPPINGS = "application.security.cors.mappings";
    public static final String APPLICATION_DATASOURCE_AVAILABLE = "application.datasource.available";
    public static final String APPLICATION_DATASOURCE_DRIVER = "application.datasource.driver";
    public static final String APPLICATION_DATASOURCE_URL = "application.datasource.url";
    public static final String APPLICATION_DATASOURCE_USERNAME = "application.datasource.username";
    public static final String APPLICATION_DATASOURCE_PASSWORD = "application.datasource.password";
    public static final String APPLICATION_DATASOURCE_IS_JNDI = "application.datasource.is-jndi";
    public static final String APPLICATION_DATASOURCE_JNDI_NAME = "application.datasource.jndi-name";
    public static final String CONFIG_JPA_PROVIDED = "config.jpa.provided";
    public static final String APPLICATION_LOCALIZATION_TIME_ZONE_ID = "application.localization.time.zone-id";
    public static final String APPLICATION_JSON_FORMAT_LOCAL_DATE = "application.json.format.local-date";
    public static final String APPLICATION_JSON_FORMAT_LOCAL_TIME = "application.json.format.local-time";
    public static final String APPLICATION_JSON_FORMAT_LOCAL_DATETIME = "application.json.format.local-datetime";

    private static final String DEFAULT_EMAIL_PATH = "$email";
    private static final String DEFAULT_PREFERRED_USERNAME_PATH = "$preferred_username";
    private static final String DEFAULT_USER_IDENTIFIER_PATH = "$sub";
    private static final String DEFAULT_REALM_ACCESS_PATH = "$realm_access.roles[*]";
    private static final String DEFAULT_DATE_FORMAT = "yyyy-MM-dd";
    private static final String DEFAULT_TIME_FORMAT = "HH:mm:ss.SSS";
    private static final String DEFAULT_DATE_TIME_FORMAT = "yyyy-MM-dd'T'HH:mm:ss.SSSX";

    public static AuthorizationSettings authorizations(ApplicationContext context) {
        var environment = context.getEnvironment();
        var decoder = context.getBean(SettingsDecoder.class);

        return Optional.of(APPLICATION_SECURITY_AUTHORIZATIONS)
                .map(environment::getProperty)
                .filter(Predicate.not(String::isBlank))
                .map(context::getResource)
                .map(SettingsFactory::getReader)
                .map(reader -> decoder.decode(reader, AuthorizationSettings.class))
                .orElseGet(() -> AuthorizationSettings.builder()
                        .policies(Collections.emptyList())
                        .build());
    }

    public static CorsSettings cors(ApplicationContext context) {
        var environment = context.getEnvironment();
        var decoder = context.getBean(SettingsDecoder.class);

        return Optional.of(APPLICATION_SECURITY_CORS_MAPPINGS)
                .map(environment::getProperty)
                .filter(Predicate.not(String::isBlank))
                .map(context::getResource)
                .map(SettingsFactory::getReader)
                .map(reader -> decoder.decode(reader, CorsSettings.class))
                .orElseGet(() -> CorsSettings.builder()
                        .mappings(Collections.emptyList())
                        .build());
    }

    public static ClaimPathSettings claimPaths(Environment environment) {
        return ClaimPathSettings.builder()
                .email(environment.getProperty(APPLICATION_SECURITY_CLAIM_PATHS_EMAIL, DEFAULT_EMAIL_PATH))
                .preferredUsername(environment.getProperty(APPLICATION_SECURITY_CLAIM_PATHS_PREFERRED_USERNAME, DEFAULT_PREFERRED_USERNAME_PATH))
                .userIdentifier(environment.getProperty(APPLICATION_SECURITY_CLAIM_PATHS_USER_IDENTIFIER, DEFAULT_USER_IDENTIFIER_PATH))
                .realmAccess(environment.getProperty(APPLICATION_SECURITY_CLAIM_PATHS_REALM_ACCESS, DEFAULT_REALM_ACCESS_PATH))
                .build();
    }

    public static DatasourceSettings datasource(Environment environment) {
        return DatasourceSettings.builder()
                .driverClassName(environment.getProperty(APPLICATION_DATASOURCE_DRIVER))
                .url(environment.getProperty(APPLICATION_DATASOURCE_URL))
                .username(environment.getProperty(APPLICATION_DATASOURCE_USERNAME))
                .password(environment.getProperty(APPLICATION_DATASOURCE_PASSWORD))
                .isJndi(environment.getProperty(APPLICATION_DATASOURCE_IS_JNDI, Boolean.class, Boolean.FALSE))
                .jndiName(environment.getProperty(APPLICATION_DATASOURCE_JNDI_NAME))
                .jpaProvided(environment.getProperty(CONFIG_JPA_PROVIDED, Boolean.class, Boolean.FALSE))
                .build();
    }

    public static LocalizationSettings localization(Environment environment) {
        return LocalizationSettings.builder()
                .zoneId(ZoneId.of(environment.getProperty(APPLICATION_LOCALIZATION_TIME_ZONE_ID, ZoneId.systemDefault().getId())))
                .localDateFormat(environment.getProperty(APPLICATION_JSON_FORMAT_LOCAL_DATE, DEFAULT_DATE_FORMAT))
                .localTimeFormat(environment.getProperty(APPLICATION_JSON_FORMAT_LOCAL_TIME, DEFAULT_TIME_FORMAT))
                .localDateTimeFormat(environment.getProperty(APPLICATION_JSON_FORMAT_LOCAL_DATETIME, DEFAULT_DATE_TIME_FORMAT))
                .build();
    }

    private static Reader getReader(Resource resource) {
        try {
            return new InputStreamReader(resource.getInputStream());
        } catch (IOException e) {
            log.error(e.getMessage(), e);
            return Reader.nullReader();
        }
    }
}
