package org.hildan.chrome.devtools.domains.layertree

import kotlin.Deprecated
import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.dom.Rect
import org.hildan.chrome.devtools.domains.layertree.events.LayerTreeEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [LayerTreeDomain.compositingReasons] command.
 */
@Serializable
public data class CompositingReasonsRequest(
  /**
   * The id of the layer for which we want to get the reasons it was composited.
   */
  public val layerId: LayerId
)

/**
 * Response type for the [LayerTreeDomain.compositingReasons] command.
 */
@Serializable
public data class CompositingReasonsResponse(
  /**
   * A list of strings specifying reasons for the given layer to become composited.
   */
  @Deprecated(message = "Deprecated in the Chrome DevTools protocol")
  public val compositingReasons: List<String>,
  /**
   * A list of strings specifying reason IDs for the given layer to become composited.
   */
  public val compositingReasonIds: List<String>
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.loadSnapshot] command.
 */
@Serializable
public data class LoadSnapshotRequest(
  /**
   * An array of tiles composing the snapshot.
   */
  public val tiles: List<PictureTile>
)

/**
 * Response type for the [LayerTreeDomain.loadSnapshot] command.
 */
@Serializable
public data class LoadSnapshotResponse(
  /**
   * The id of the snapshot.
   */
  public val snapshotId: SnapshotId
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.makeSnapshot] command.
 */
@Serializable
public data class MakeSnapshotRequest(
  /**
   * The id of the layer.
   */
  public val layerId: LayerId
)

/**
 * Response type for the [LayerTreeDomain.makeSnapshot] command.
 */
@Serializable
public data class MakeSnapshotResponse(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.profileSnapshot] command.
 */
@Serializable
public data class ProfileSnapshotRequest(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId,
  /**
   * The maximum number of times to replay the snapshot (1, if not specified).
   */
  public val minRepeatCount: Int? = null,
  /**
   * The minimum duration (in seconds) to replay the snapshot.
   */
  public val minDuration: Double? = null,
  /**
   * The clip rectangle to apply when replaying the snapshot.
   */
  public val clipRect: Rect? = null
)

/**
 * Response type for the [LayerTreeDomain.profileSnapshot] command.
 */
@Serializable
public data class ProfileSnapshotResponse(
  /**
   * The array of paint profiles, one per run.
   */
  public val timings: List<PaintProfile>
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.releaseSnapshot] command.
 */
@Serializable
public data class ReleaseSnapshotRequest(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.replaySnapshot] command.
 */
@Serializable
public data class ReplaySnapshotRequest(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId,
  /**
   * The first step to replay from (replay from the very start if not specified).
   */
  public val fromStep: Int? = null,
  /**
   * The last step to replay to (replay till the end if not specified).
   */
  public val toStep: Int? = null,
  /**
   * The scale to apply while replaying (defaults to 1).
   */
  public val scale: Double? = null
)

/**
 * Response type for the [LayerTreeDomain.replaySnapshot] command.
 */
@Serializable
public data class ReplaySnapshotResponse(
  /**
   * A data: URL for resulting image.
   */
  public val dataURL: String
)

/**
 * Request object containing input parameters for the [LayerTreeDomain.snapshotCommandLog] command.
 */
@Serializable
public data class SnapshotCommandLogRequest(
  /**
   * The id of the layer snapshot.
   */
  public val snapshotId: SnapshotId
)

/**
 * Response type for the [LayerTreeDomain.snapshotCommandLog] command.
 */
@Serializable
public data class SnapshotCommandLogResponse(
  /**
   * The array of canvas function calls.
   */
  public val commandLog: List<JsonElement>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree)
 */
@ExperimentalChromeApi
public class LayerTreeDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out LayerTreeEvent>> =
      mapOf(
      "LayerTree.layerPainted" to serializer<LayerTreeEvent.LayerPaintedEvent>(),
      "LayerTree.layerTreeDidChange" to serializer<LayerTreeEvent.LayerTreeDidChangeEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<LayerTreeEvent> = session.events(deserializersByEventName)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#event-layerPainted)
   */
  public fun layerPainted(): Flow<LayerTreeEvent.LayerPaintedEvent> =
      session.events("LayerTree.layerPainted")

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#event-layerTreeDidChange)
   */
  public fun layerTreeDidChange(): Flow<LayerTreeEvent.LayerTreeDidChangeEvent> =
      session.events("LayerTree.layerTreeDidChange")

  /**
   * Provides the reasons why the given layer was composited.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-compositingReasons)
   */
  public suspend fun compositingReasons(input: CompositingReasonsRequest):
      CompositingReasonsResponse = session.request("LayerTree.compositingReasons", input)

  /**
   * Disables compositing tree inspection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("LayerTree.disable", Unit)

  /**
   * Enables compositing tree inspection.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("LayerTree.enable", Unit)

  /**
   * Returns the snapshot identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-loadSnapshot)
   */
  public suspend fun loadSnapshot(input: LoadSnapshotRequest): LoadSnapshotResponse =
      session.request("LayerTree.loadSnapshot", input)

  /**
   * Returns the layer snapshot identifier.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-makeSnapshot)
   */
  public suspend fun makeSnapshot(input: MakeSnapshotRequest): MakeSnapshotResponse =
      session.request("LayerTree.makeSnapshot", input)

  /**
   *
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-profileSnapshot)
   */
  public suspend fun profileSnapshot(input: ProfileSnapshotRequest): ProfileSnapshotResponse =
      session.request("LayerTree.profileSnapshot", input)

  /**
   * Releases layer snapshot captured by the back-end.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-releaseSnapshot)
   */
  public suspend fun releaseSnapshot(input: ReleaseSnapshotRequest): Unit =
      session.request("LayerTree.releaseSnapshot", input)

  /**
   * Replays the layer snapshot and returns the resulting bitmap.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-replaySnapshot)
   */
  public suspend fun replaySnapshot(input: ReplaySnapshotRequest): ReplaySnapshotResponse =
      session.request("LayerTree.replaySnapshot", input)

  /**
   * Replays the layer snapshot and returns canvas log.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/LayerTree/#method-snapshotCommandLog)
   */
  public suspend fun snapshotCommandLog(input: SnapshotCommandLogRequest):
      SnapshotCommandLogResponse = session.request("LayerTree.snapshotCommandLog", input)
}
