package org.hildan.chrome.devtools.domains.media

import kotlin.String
import kotlin.Unit
import kotlin.collections.Map
import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.DeserializationStrategy
import kotlinx.serialization.serializer
import org.hildan.chrome.devtools.domains.media.events.MediaEvent
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * This domain allows detailed inspection of media elements
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media)
 */
@ExperimentalChromeApi
public class MediaDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Mapping between events and their deserializer.
   */
  private val deserializersByEventName: Map<String, DeserializationStrategy<out MediaEvent>> =
      mapOf(
      "Media.playerPropertiesChanged" to serializer<MediaEvent.PlayerPropertiesChangedEvent>(),
      "Media.playerEventsAdded" to serializer<MediaEvent.PlayerEventsAddedEvent>(),
      "Media.playerMessagesLogged" to serializer<MediaEvent.PlayerMessagesLoggedEvent>(),
      "Media.playerErrorsRaised" to serializer<MediaEvent.PlayerErrorsRaisedEvent>(),
      "Media.playersCreated" to serializer<MediaEvent.PlayersCreatedEvent>(),
      )

  /**
   * Subscribes to all events related to this domain.
   */
  public fun events(): Flow<MediaEvent> = session.events(deserializersByEventName)

  /**
   * This can be called multiple times, and can be used to set / override /
   * remove player properties. A null propValue indicates removal.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerPropertiesChanged)
   */
  public fun playerPropertiesChanged(): Flow<MediaEvent.PlayerPropertiesChangedEvent> =
      session.events("Media.playerPropertiesChanged")

  /**
   * Send events as a list, allowing them to be batched on the browser for less
   * congestion. If batched, events must ALWAYS be in chronological order.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerEventsAdded)
   */
  public fun playerEventsAdded(): Flow<MediaEvent.PlayerEventsAddedEvent> =
      session.events("Media.playerEventsAdded")

  /**
   * Send a list of any messages that need to be delivered.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerMessagesLogged)
   */
  public fun playerMessagesLogged(): Flow<MediaEvent.PlayerMessagesLoggedEvent> =
      session.events("Media.playerMessagesLogged")

  /**
   * Send a list of any errors that need to be delivered.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playerErrorsRaised)
   */
  public fun playerErrorsRaised(): Flow<MediaEvent.PlayerErrorsRaisedEvent> =
      session.events("Media.playerErrorsRaised")

  /**
   * Called whenever a player is created, or when a new agent joins and recieves
   * a list of active players. If an agent is restored, it will recieve the full
   * list of player ids and all events again.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#event-playersCreated)
   */
  public fun playersCreated(): Flow<MediaEvent.PlayersCreatedEvent> =
      session.events("Media.playersCreated")

  /**
   * Enables the Media domain
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Media.enable", Unit)

  /**
   * Disables the Media domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Media/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Media.disable", Unit)
}
