package org.hildan.chrome.devtools.domains.accessibility

import kotlin.Boolean
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.dom.BackendNodeId
import org.hildan.chrome.devtools.domains.dom.NodeId
import org.hildan.chrome.devtools.domains.runtime.RemoteObjectId
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [AccessibilityDomain.getPartialAXTree]
 * command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetPartialAXTreeRequest(
  /**
   * Identifier of the node to get the partial accessibility tree for.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node to get the partial accessibility tree for.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper to get the partial accessibility tree for.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Whether to fetch this nodes ancestors, siblings and children. Defaults to true.
   */
  public val fetchRelatives: Boolean? = null
)

/**
 * Response type for the [AccessibilityDomain.getPartialAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetPartialAXTreeResponse(
  /**
   * The `Accessibility.AXNode` for this DOM node, if it exists, plus its ancestors, siblings and
   * children, if requested.
   */
  public val nodes: List<AXNode>
)

/**
 * Request object containing input parameters for the [AccessibilityDomain.getFullAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFullAXTreeRequest(
  /**
   * The maximum depth at which descendants of the root node should be retrieved.
   * If omitted, the full tree is returned.
   */
  public val max_depth: Int? = null
)

/**
 * Response type for the [AccessibilityDomain.getFullAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetFullAXTreeResponse(
  public val nodes: List<AXNode>
)

/**
 * Request object containing input parameters for the [AccessibilityDomain.getChildAXNodes] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetChildAXNodesRequest(
  public val id: AXNodeId
)

/**
 * Response type for the [AccessibilityDomain.getChildAXNodes] command.
 */
@Serializable
@ExperimentalChromeApi
public data class GetChildAXNodesResponse(
  public val nodes: List<AXNode>
)

/**
 * Request object containing input parameters for the [AccessibilityDomain.queryAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class QueryAXTreeRequest(
  /**
   * Identifier of the node for the root to query.
   */
  public val nodeId: NodeId? = null,
  /**
   * Identifier of the backend node for the root to query.
   */
  public val backendNodeId: BackendNodeId? = null,
  /**
   * JavaScript object id of the node wrapper for the root to query.
   */
  public val objectId: RemoteObjectId? = null,
  /**
   * Find nodes with this computed name.
   */
  public val accessibleName: String? = null,
  /**
   * Find nodes with this computed role.
   */
  public val role: String? = null
)

/**
 * Response type for the [AccessibilityDomain.queryAXTree] command.
 */
@Serializable
@ExperimentalChromeApi
public data class QueryAXTreeResponse(
  /**
   * A list of `Accessibility.AXNode` matching the specified attributes,
   * including nodes that are ignored for accessibility.
   */
  public val nodes: List<AXNode>
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility)
 */
@ExperimentalChromeApi
public class AccessibilityDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Disables the accessibility domain.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-disable)
   */
  public suspend fun disable(): Unit = session.request("Accessibility.disable", Unit)

  /**
   * Enables the accessibility domain which causes `AXNodeId`s to remain consistent between method
   * calls.
   * This turns on accessibility for the page, which can impact performance until accessibility is
   * disabled.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-enable)
   */
  public suspend fun enable(): Unit = session.request("Accessibility.enable", Unit)

  /**
   * Fetches the accessibility node and partial accessibility tree for this DOM node, if it exists.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getPartialAXTree)
   */
  @ExperimentalChromeApi
  public suspend fun getPartialAXTree(input: GetPartialAXTreeRequest): GetPartialAXTreeResponse =
      session.request("Accessibility.getPartialAXTree", input)

  /**
   * Fetches the entire accessibility tree for the root Document
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getFullAXTree)
   */
  @ExperimentalChromeApi
  public suspend fun getFullAXTree(input: GetFullAXTreeRequest): GetFullAXTreeResponse =
      session.request("Accessibility.getFullAXTree", input)

  /**
   * Fetches a particular accessibility node by AXNodeId.
   * Requires `enable()` to have been called previously.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-getChildAXNodes)
   */
  @ExperimentalChromeApi
  public suspend fun getChildAXNodes(input: GetChildAXNodesRequest): GetChildAXNodesResponse =
      session.request("Accessibility.getChildAXNodes", input)

  /**
   * Query a DOM node's accessibility subtree for accessible name and role.
   * This command computes the name and role for all nodes in the subtree, including those that are
   * ignored for accessibility, and returns those that mactch the specified name and role. If no DOM
   * node is specified, or the DOM node does not exist, the command returns an error. If neither
   * `accessibleName` or `role` is specified, it returns all the accessibility nodes in the subtree.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/Accessibility/#method-queryAXTree)
   */
  @ExperimentalChromeApi
  public suspend fun queryAXTree(input: QueryAXTreeRequest): QueryAXTreeResponse =
      session.request("Accessibility.queryAXTree", input)
}
