package org.hildan.chrome.devtools.domains.cachestorage

import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.collections.List
import kotlinx.serialization.Serializable

/**
 * Unique identifier of the Cache object.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#type-CacheId)
 */
public typealias CacheId = String

/**
 * type of HTTP response cached
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#type-CachedResponseType)
 */
@Serializable
public enum class CachedResponseType {
  basic,
  cors,
  default,
  error,
  opaqueResponse,
  opaqueRedirect,
}

/**
 * Data entry.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#type-DataEntry)
 */
@Serializable
public data class DataEntry(
  /**
   * Request URL.
   */
  public val requestURL: String,
  /**
   * Request method.
   */
  public val requestMethod: String,
  /**
   * Request headers
   */
  public val requestHeaders: List<Header>,
  /**
   * Number of seconds since epoch.
   */
  public val responseTime: Double,
  /**
   * HTTP response status code.
   */
  public val responseStatus: Int,
  /**
   * HTTP response status text.
   */
  public val responseStatusText: String,
  /**
   * HTTP response type
   */
  public val responseType: CachedResponseType,
  /**
   * Response headers
   */
  public val responseHeaders: List<Header>
)

/**
 * Cache identifier.
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#type-Cache)
 */
@Serializable
public data class Cache(
  /**
   * An opaque unique id of the cache.
   */
  public val cacheId: CacheId,
  /**
   * Security origin of the cache.
   */
  public val securityOrigin: String,
  /**
   * The name of the cache.
   */
  public val cacheName: String
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#type-Header)
 */
@Serializable
public data class Header(
  public val name: String,
  public val value: String
)

/**
 * Cached response
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#type-CachedResponse)
 */
@Serializable
public data class CachedResponse(
  /**
   * Entry content, base64-encoded. (Encoded as a base64 string when passed over JSON)
   */
  public val body: String
)
