package org.hildan.chrome.devtools.domains.webaudio.events

import kotlin.Double
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.domains.webaudio.AudioListener
import org.hildan.chrome.devtools.domains.webaudio.AudioNode
import org.hildan.chrome.devtools.domains.webaudio.AudioParam
import org.hildan.chrome.devtools.domains.webaudio.BaseAudioContext
import org.hildan.chrome.devtools.domains.webaudio.GraphObjectId

@Serializable
public sealed class WebAudioEvent {
  /**
   * Notifies that a new BaseAudioContext has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextCreated)
   */
  @Serializable
  public data class ContextCreatedEvent(
    public val context: BaseAudioContext
  ) : WebAudioEvent()

  /**
   * Notifies that an existing BaseAudioContext will be destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextWillBeDestroyed)
   */
  @Serializable
  public data class ContextWillBeDestroyedEvent(
    public val contextId: GraphObjectId
  ) : WebAudioEvent()

  /**
   * Notifies that existing BaseAudioContext has changed some properties (id stays the same)..
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-contextChanged)
   */
  @Serializable
  public data class ContextChangedEvent(
    public val context: BaseAudioContext
  ) : WebAudioEvent()

  /**
   * Notifies that the construction of an AudioListener has finished.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioListenerCreated)
   */
  @Serializable
  public data class AudioListenerCreatedEvent(
    public val listener: AudioListener
  ) : WebAudioEvent()

  /**
   * Notifies that a new AudioListener has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioListenerWillBeDestroyed)
   */
  @Serializable
  public data class AudioListenerWillBeDestroyedEvent(
    public val contextId: GraphObjectId,
    public val listenerId: GraphObjectId
  ) : WebAudioEvent()

  /**
   * Notifies that a new AudioNode has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioNodeCreated)
   */
  @Serializable
  public data class AudioNodeCreatedEvent(
    public val node: AudioNode
  ) : WebAudioEvent()

  /**
   * Notifies that an existing AudioNode has been destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioNodeWillBeDestroyed)
   */
  @Serializable
  public data class AudioNodeWillBeDestroyedEvent(
    public val contextId: GraphObjectId,
    public val nodeId: GraphObjectId
  ) : WebAudioEvent()

  /**
   * Notifies that a new AudioParam has been created.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioParamCreated)
   */
  @Serializable
  public data class AudioParamCreatedEvent(
    public val `param`: AudioParam
  ) : WebAudioEvent()

  /**
   * Notifies that an existing AudioParam has been destroyed.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-audioParamWillBeDestroyed)
   */
  @Serializable
  public data class AudioParamWillBeDestroyedEvent(
    public val contextId: GraphObjectId,
    public val nodeId: GraphObjectId,
    public val paramId: GraphObjectId
  ) : WebAudioEvent()

  /**
   * Notifies that two AudioNodes are connected.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodesConnected)
   */
  @Serializable
  public data class NodesConnectedEvent(
    public val contextId: GraphObjectId,
    public val sourceId: GraphObjectId,
    public val destinationId: GraphObjectId,
    public val sourceOutputIndex: Double? = null,
    public val destinationInputIndex: Double? = null
  ) : WebAudioEvent()

  /**
   * Notifies that AudioNodes are disconnected. The destination can be null, and it means all the
   * outgoing connections from the source are disconnected.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodesDisconnected)
   */
  @Serializable
  public data class NodesDisconnectedEvent(
    public val contextId: GraphObjectId,
    public val sourceId: GraphObjectId,
    public val destinationId: GraphObjectId,
    public val sourceOutputIndex: Double? = null,
    public val destinationInputIndex: Double? = null
  ) : WebAudioEvent()

  /**
   * Notifies that an AudioNode is connected to an AudioParam.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodeParamConnected)
   */
  @Serializable
  public data class NodeParamConnectedEvent(
    public val contextId: GraphObjectId,
    public val sourceId: GraphObjectId,
    public val destinationId: GraphObjectId,
    public val sourceOutputIndex: Double? = null
  ) : WebAudioEvent()

  /**
   * Notifies that an AudioNode is disconnected to an AudioParam.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/WebAudio/#event-nodeParamDisconnected)
   */
  @Serializable
  public data class NodeParamDisconnectedEvent(
    public val contextId: GraphObjectId,
    public val sourceId: GraphObjectId,
    public val destinationId: GraphObjectId,
    public val sourceOutputIndex: Double? = null
  ) : WebAudioEvent()
}
