package org.hildan.chrome.devtools.domains.cachestorage

import kotlin.Double
import kotlin.Int
import kotlin.String
import kotlin.Unit
import kotlin.collections.List
import kotlinx.serialization.Serializable
import org.hildan.chrome.devtools.protocol.ChromeDPSession
import org.hildan.chrome.devtools.protocol.ExperimentalChromeApi

/**
 * Request object containing input parameters for the [CacheStorageDomain.deleteCache] command.
 */
@Serializable
public data class DeleteCacheRequest(
  /**
   * Id of cache for deletion.
   */
  public val cacheId: CacheId
)

/**
 * Request object containing input parameters for the [CacheStorageDomain.deleteEntry] command.
 */
@Serializable
public data class DeleteEntryRequest(
  /**
   * Id of cache where the entry will be deleted.
   */
  public val cacheId: CacheId,
  /**
   * URL spec of the request.
   */
  public val request: String
)

/**
 * Request object containing input parameters for the [CacheStorageDomain.requestCacheNames]
 * command.
 */
@Serializable
public data class RequestCacheNamesRequest(
  /**
   * Security origin.
   */
  public val securityOrigin: String
)

/**
 * Response type for the [CacheStorageDomain.requestCacheNames] command.
 */
@Serializable
public data class RequestCacheNamesResponse(
  /**
   * Caches for the security origin.
   */
  public val caches: List<Cache>
)

/**
 * Request object containing input parameters for the [CacheStorageDomain.requestCachedResponse]
 * command.
 */
@Serializable
public data class RequestCachedResponseRequest(
  /**
   * Id of cache that contains the entry.
   */
  public val cacheId: CacheId,
  /**
   * URL spec of the request.
   */
  public val requestURL: String,
  /**
   * headers of the request.
   */
  public val requestHeaders: List<Header>
)

/**
 * Response type for the [CacheStorageDomain.requestCachedResponse] command.
 */
@Serializable
public data class RequestCachedResponseResponse(
  /**
   * Response read from the cache.
   */
  public val response: CachedResponse
)

/**
 * Request object containing input parameters for the [CacheStorageDomain.requestEntries] command.
 */
@Serializable
public data class RequestEntriesRequest(
  /**
   * ID of cache to get entries from.
   */
  public val cacheId: CacheId,
  /**
   * Number of records to skip.
   */
  public val skipCount: Int? = null,
  /**
   * Number of records to fetch.
   */
  public val pageSize: Int? = null,
  /**
   * If present, only return the entries containing this substring in the path
   */
  public val pathFilter: String? = null
)

/**
 * Response type for the [CacheStorageDomain.requestEntries] command.
 */
@Serializable
public data class RequestEntriesResponse(
  /**
   * Array of object store data entries.
   */
  public val cacheDataEntries: List<DataEntry>,
  /**
   * Count of returned entries from this storage. If pathFilter is empty, it
   * is the count of all entries from this storage.
   */
  public val returnCount: Double
)

/**
 *
 *
 * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage)
 */
@ExperimentalChromeApi
public class CacheStorageDomain internal constructor(
  private val session: ChromeDPSession
) {
  /**
   * Deletes a cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-deleteCache)
   */
  public suspend fun deleteCache(input: DeleteCacheRequest): Unit =
      session.request("CacheStorage.deleteCache", input)

  /**
   * Deletes a cache entry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-deleteEntry)
   */
  public suspend fun deleteEntry(input: DeleteEntryRequest): Unit =
      session.request("CacheStorage.deleteEntry", input)

  /**
   * Requests cache names.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestCacheNames)
   */
  public suspend fun requestCacheNames(input: RequestCacheNamesRequest): RequestCacheNamesResponse =
      session.request("CacheStorage.requestCacheNames", input)

  /**
   * Fetches cache entry.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestCachedResponse)
   */
  public suspend fun requestCachedResponse(input: RequestCachedResponseRequest):
      RequestCachedResponseResponse = session.request("CacheStorage.requestCachedResponse", input)

  /**
   * Requests data from cache.
   *
   * [Official doc](https://chromedevtools.github.io/devtools-protocol/tot/CacheStorage/#method-requestEntries)
   */
  public suspend fun requestEntries(input: RequestEntriesRequest): RequestEntriesResponse =
      session.request("CacheStorage.requestEntries", input)
}
